#include "blackbox_proxy.h"

#include "blackbox_client.h"
#include "cache.h"
#include "get_ticket.h"

#include <library/cpp/ipv6_address/ipv6_address.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/tvmauth/checked_service_ticket.h>
#include <library/cpp/tvmknife/output.h>

#include <util/network/interface.h>
#include <util/stream/output.h>
#include <util/string/builder.h>

#include <vector>

namespace NPassport::NTvmknife::NBbProxy {
    TCredential::TCredential(const TString& blackbox, const TString& userip)
        : Blackbox_(blackbox)
        , BlackboxTvmId_(BbTvmId(Blackbox_))
        , Userip_(userip)
    {
    }

    TString TCredential::GetUserIp() const {
        return Userip_ ? Userip_ : GetSelfIp();
    }

    TOAuthCredential::TOAuthCredential(const TString& blackbox, const TBbClient::TDebugUTParams& params)
        : TCredential(blackbox)
        , Params_(params)
    {
    }

    TString TOAuthCredential::TryGetFromCache() const {
        return TCache::Get().GetUsrTicket(
            TCache::TUsrTick::FromOAuth(BlackboxTvmId_,
                                        Params_.OauthToken,
                                        Params_.AddScopeSessionid,
                                        Params_.AddScopeSessguard));
    }

    void TOAuthCredential::PutToCache(const TString& ticket) const {
        TCache::Get().SetUserTicket(
            TCache::TUsrTick::FromOAuth(BlackboxTvmId_,
                                        Params_.OauthToken,
                                        Params_.AddScopeSessionid,
                                        Params_.AddScopeSessguard),
            ticket);
    }

    TBbClient::TRequest TOAuthCredential::CreateRequest() const {
        return TBbClient::CreateDebugUserTicketRequest(Params_);
    }

    TString GetUserTicket(const TCredential& cred, TStringBuf serviceTicket) {
        serviceTicket.ChopSuffix("\n");
        TString res = TUserTicket::GetTicketFromResponse(
            TBbClient::DoRequest(cred.GetBlackbox(),
                                 cred.CreateRequest(),
                                 serviceTicket));
        res.push_back('\n');
        cred.PutToCache(res);
        return res;
    }

    using TMapId = std::vector<std::pair<TStringBuf, TStringBuf>>;
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    static TMapId BB_TVM_ID = {
        {"blackbox.yandex.net", NTvmAuth::NBlackboxTvmId::Prod},
        {"blackbox-rc.yandex.net", NTvmAuth::NBlackboxTvmId::Prod},
        {"blackbox-mail.yandex.net", NTvmAuth::NBlackboxTvmId::Prod},

        {"blackbox-test.yandex.net", NTvmAuth::NBlackboxTvmId::Test},
        {"pass-test.yandex.ru", NTvmAuth::NBlackboxTvmId::Test},

        {"blackbox.yandex-team.ru", NTvmAuth::NBlackboxTvmId::ProdYateam},

        {"blackbox-test.yandex-team.ru", NTvmAuth::NBlackboxTvmId::TestYateam},

        {"blackbox-stress.yandex.net", NTvmAuth::NBlackboxTvmId::Stress},
        {"pass-stress-f1.sezam.yandex.net", NTvmAuth::NBlackboxTvmId::Stress},
        {"pass-stress-i1.sezam.yandex.net", NTvmAuth::NBlackboxTvmId::Stress},
        {"pass-stress-m1.sezam.yandex.net", NTvmAuth::NBlackboxTvmId::Stress},
        {"pass-stress-s1.sezam.yandex.net", NTvmAuth::NBlackboxTvmId::Stress},

        {"blackbox-mimino.yandex.net", NTvmAuth::NBlackboxTvmId::Mimino},
    };

    TString BbTvmId(const TString& blackbox) {
        const TStringBuf host = GetOnlyHost(blackbox);
        auto it = std::find_if(BB_TVM_ID.begin(),
                               BB_TVM_ID.end(),
                               [host](const std::pair<TStringBuf, TStringBuf>& pair) {
                                   return pair.first == host;
                               });

        if (it == BB_TVM_ID.end()) {
            ythrow yexception() << "Unknown blackbox host '" << blackbox << "'." << Endl
                                << "Please choose one of:" << Endl
                                << BB_TVM_ID;
        }

        return TString(it->second);
    }

    TString ListBlackboxes() {
        return TStringBuilder() << BB_TVM_ID;
    }

    std::vector<TString> SuggestBlackboxes() {
        return {
            "https://blackbox-mimino.yandex.net",
            "https://blackbox-stress.yandex.net",
            "https://blackbox-test.yandex.net",
            "https://blackbox-test.yandex-team.ru",
            "https://blackbox.yandex.net",
            "https://blackbox.yandex-team.ru",
        };
    }

    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TString USERIP;

    TString GetSelfIp() {
        if (USERIP) {
            return USERIP;
        }

        NAddr::TNetworkInterfaceList list = NAddr::GetNetworkInterfaces();
        for (auto it = list.begin(); it != list.end();) {
            if (IsLoopback(*it->Address)) {
                list.erase(it++);
            } else {
                ++it;
            }
        }

        std::sort(list.begin(),
                  list.end(),
                  [](const NAddr::TNetworkInterface& l, const NAddr::TNetworkInterface& r) -> bool {
                      TIpv6Address la(*l.Address);
                      TIpv6Address ra(*r.Address);

                      return la.Type() == ra.Type()
                                 ? la.ScopeId() < ra.ScopeId() // global < host < link
                                 : la.Type() == TIpv6Address::TIpType::Ipv6;
                  });

        USERIP = list.empty() ? "127.0.0.1" : PrintHost(*list[0].Address);
        ::NTvmknife::NOutput::Out() << "Detected userip: " << USERIP << Endl;
        return USERIP;
    }

    void AddBbTvmId(const TStringBuf blackbox) {
        BB_TVM_ID.push_back({blackbox, "0"});
    }
}

template <>
void Out<NPassport::NTvmknife::NBbProxy::TMapId>(IOutputStream& o, const NPassport::NTvmknife::NBbProxy::TMapId& value) {
    for (const auto& pair : value) {
        o << pair.first << " -> tvm_id=" << pair.second << Endl;
    }
}
