#include "cache.h"

#include "tvm_client.h"

#include <passport/infra/tools/tvmknife/src/proto/tickets.pb.h>

#include <library/cpp/tvmknife/output.h>

#include <util/datetime/base.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

namespace NPassport::NTvmknife {
    static const TDuration LIFETIME_OF_PUBLIC_KEYS = TDuration::Days(1);
    static const TDuration LIFETIME_OF_USR_TICKETS = TDuration::Seconds(150);
    static const TString PUBLIC_KEYS_FILENAME = "public_keys";
    static const TString USER_TICKETS_FILENAME = "user_tickets";

    TString TCache::GetPublicKeysPath() {
        return CacheDir_ + PUBLIC_KEYS_FILENAME;
    }

    TString TCache::GetUserTicketsPath() {
        return CacheDir_ + USER_TICKETS_FILENAME;
    }

    void TCache::Reset() {
        PublicKeys_.clear();
        UsrTickets_.clear();
        ::NTvmknife::TCache::Reset();
    }

    void TCache::WriteUsrTicket(const TCache::TUsrTick& t) {
        UsrTickets_.insert(t);

        TString toDisk = SerializeUsrTickets(UsrTickets_);
        if (toDisk) {
            WriteToDisk(GetUserTicketsPath(), toDisk);
        }
    }

    TCache::TCache() = default;

    TCache& TCache::Get() {
        static TCache ins;
        return ins;
    }

    TString TCache::GetPublicKeys() {
        if (PublicKeys_.empty()) {
            PublicKeys_ = ReadFromDisk(GetPublicKeysPath(), LIFETIME_OF_PUBLIC_KEYS);
        }

        return PublicKeys_;
    }

    void TCache::SetPublicKeys(const TString& keys) {
        PublicKeys_ = keys;
        WriteToDisk(GetPublicKeysPath(), keys);
    }

    TString TCache::GetUsrTicket(const TCache::TUsrTick& t) {
        if (UsrTickets_.empty()) {
            UsrTickets_ = ParseUsrTickets(ReadFromDisk(GetUserTicketsPath(), LIFETIME_OF_USR_TICKETS));
        }

        auto it = UsrTickets_.find(t);
        return it == UsrTickets_.end() ? TString() : it->Body;
    }

    void TCache::SetUserTicket(TCache::TUsrTick&& t, const TString& body) {
        t.Body = body;
        WriteUsrTicket(t);
    }

    TString TCache::SerializeUsrTickets(const std::set<TCache::TUsrTick>& ticks) {
        tickets::UsrRows r;
        for (const TUsrTick& it : ticks) {
            tickets::UsrRow* row = r.add_rows();
            row->set_token(it.Token);
            row->set_ticket(it.Body);
            row->set_ts(it.Ts);
            row->set_bbtvmid(it.BbTvmId);
            row->set_add_scope_sessionid(it.AddScopeSessionId);
            row->set_add_scope_sessguard(it.AddScopeSessguard);
        }

        return r.SerializeAsString();
    }

    std::set<TCache::TUsrTick> TCache::ParseUsrTickets(const TString& ticks) {
        tickets::UsrRows r;
        if (!r.ParseFromString(ticks)) {
            return {};
        }

        std::set<TCache::TUsrTick> res;
        for (int idx = 0; idx < r.rows_size(); ++idx) {
            time_t ts = r.rows(idx).Getts();
            if (TInstant::Now() - TInstant::Seconds(ts) > LIFETIME_OF_USR_TICKETS) {
                continue;
            }

            TUsrTick k;
            k.Token = r.rows(idx).Gettoken();
            k.Body = r.rows(idx).Getticket();
            k.Ts = ts;
            k.BbTvmId = r.rows(idx).GetbbTvmId();
            k.AddScopeSessionId = r.rows(idx).Getadd_scope_sessionid();
            k.AddScopeSessguard = r.rows(idx).Getadd_scope_sessguard();

            res.insert(k);
        }

        return res;
    }

    void TCache::SetCacheDir(const TString& dir) {
        CacheDir_ = dir;
        Reset();
    }
}
