#include "info.h"

#include "tvm_client.h"

#include <passport/infra/libs/cpp/json/reader.h>

#include <library/cpp/colorizer/colors.h>

#include <util/stream/str.h>
#include <util/string/cast.h>

namespace NPassport::NTvmknife {
    TInfo::TTvmInfo TInfo::GetInfoFromTvmResponse(const TString& response) {
        if (!response) {
            return {};
        }

        using namespace NPassport;
        try {
            rapidjson::Document doc;
            Y_ENSURE(NJson::TReader::DocumentAsObject(response, doc));

            TString status;
            if (NJson::TReader::MemberAsString(doc, "status", status) && status == "error") {
                return {};
            }

            TTvmInfo res;
            Y_ENSURE(NJson::TReader::MemberAsString(doc, "name", res.Description));

            NJson::TReader::MemberAsUInt64(doc, "abc_service_id", res.AbcId);

            time_t time = 0;
            if (NJson::TReader::MemberAsInt64(doc, "deletion_time", time)) {
                res.DeletedAt = TInstant::Seconds(time);
            }

            NJson::TReader::MemberAsString(doc, "vault_link", res.VaultLink);

            Y_ENSURE(NJson::TReader::MemberAsUInt64(doc, "creator_uid", res.CreatorUid));

            return res;
        } catch (const std::exception& e) {
            ythrow yexception() << "Failed to get info from TVM response: " << e.what() << Endl
                                << response << Endl;
        }
    }

    TString TInfo::GetInfoImpl(const TString& id,
                               const TInfo::TTvmInfo& tvmInfo,
                               TAdd add,
                               EColored colored) {
        TStringStream s;

        if (!tvmInfo.Description) {
            s << "Client with id=" << id << " not found" << Endl;
            return s.Str();
        }

        s << "Client id (tvm id): " << id;
        if (tvmInfo.DeletedAt) {
            NColorizer::TColors c(colored == EColored::True);
            s << c.YellowColor() << " - deleted at: " << FormatLocalUpToSeconds(*tvmInfo.DeletedAt) << c.OldColor();
        }
        s << Endl;
        s << "Client description: " << tvmInfo.Description << Endl;

        if (tvmInfo.AbcId > 0) {
            s << "ABC service: https://abc.yandex-team.ru/services/" << tvmInfo.AbcId << "/" << Endl;
        } else {
            s << "Client is not linked with ABC service" << Endl;
        }

        if (add.Yav) {
            s << "Tvm secret in yav: " << (tvmInfo.VaultLink ? tvmInfo.VaultLink : "NULL") << Endl;
        }

        if (add.Creator) {
            s << "Creator: https://staff-api.yandex-team.ru/v3/persons?uid=" << tvmInfo.CreatorUid << Endl;
        }

        return s.Str();
    }

    TString TInfo::GetInfo(const TString& id, TAdd add) {
        TTvmInfo tvm = GetInfoFromTvmResponse(TTvmClient::Info(id));
        return GetInfoImpl(id, tvm, add);
    }
}
