#include "menu.h"

#include "aliases.h"
#include "blackbox_proxy.h"
#include "info.h"
#include "oauth_clients.h"
#include "parse_anyway.h"
#include "parse_keys.h"
#include "ssh_utils.h"
#include "terminal_attrs.h"
#include "tvm_proxy.h"
#include "unittest.h"

#include <passport/infra/libs/cpp/utils/string/split.h>

#include <library/cpp/getopt/small/completer.h>
#include <library/cpp/tvmknife/output.h>

#include <util/generic/ptr.h>
#include <util/network/socket.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/system/fs.h>

namespace NPassport::NTvmknife {
    static void AddDefaultOpts(NLastGetopt::TOpts& opts, int freeArgs = 0) {
        opts.AddHelpOption();
        opts.SetFreeArgsNum(freeArgs);
        opts.AddLongOption('q', "quiet", "skip all debug messages")
            .StoreTrue(&::NTvmknife::NOutput::GetQuiteFlag())
            .NoArgument();
    }

    static NLastGetopt::NComp::ICompleterPtr Choises(const std::vector<TString>& choises) {
        TVector<NLastGetopt::NComp::TChoice> vec;
        vec.reserve(choises.size());

        for (const TString& s : choises) {
            vec.push_back(s);
        }

        return NLastGetopt::NComp::Choice(std::move(vec));
    }

    TConfidentialValue::TConfidentialValue(NLastGetopt::TOpts& opts,
                                           char shortKey,
                                           const TString& longKey,
                                           const TString& help)
        : LongKey_(longKey)
    {
        opts.AddLongOption(shortKey,
                           longKey,
                           help + ".\nMay be read from STDIN (by default) or from specified file")
            .StoreResult(&Path_)
            .OptionalValue("STDIN", "FROM");
    }

    bool TConfidentialValue::IsDefault() const {
        return Path_.empty();
    }

    const TString& TConfidentialValue::Get() {
        if (Value_.empty()) {
            Value_ = ReadValue();
        }

        return Value_;
    }

    TString TConfidentialValue::ReadValue() const {
        if (Path_ && Path_ != "STDIN") {
            return ReadFromFile(Path_);
        }

        return ReadFromStdin(LongKey_);
    }

    TString TConfidentialValue::ReadFromFile(const TString& path) {
        TString res;

        TFileInput file(path);
        if (!file.ReadLine(res)) {
            ythrow yexception() << "File \"" << path << "\" is empty";
        }

        return res;
    }

    TString TConfidentialValue::ReadFromStdin(const TString& longKey) {
        if (!IsStdinEmpty()) {
            return Cin.ReadLine();
        }

        TString res;

        while (res.empty()) {
            Cerr << "Enter '" << longKey << "': ";

            res = ReadFromStdinImpl();
            Cerr << Endl;
        }

        return res;
    }

    bool TConfidentialValue::IsStdinEmpty() {
        pollfd fd{};
#ifdef _unix_
        fd.fd = STDIN_FILENO;
#else
        fd.fd = _fileno(stdin);
#endif

        const int readyFdCount = poll(&fd, 1, 0);

        return readyFdCount == 0;
    }

#ifdef _unix_
    TString TConfidentialValue::ReadFromStdinImpl() {
        try {
            TTerminalAttrs attrs;

            /* Turn echoing off and fail if we can't. */
            termios newAttrs = attrs.GetOldAttrs();
            newAttrs.c_lflag &= ~ECHO;
            TTerminalAttrs::TrySetNewAttrs(newAttrs);

            /* Read the password. */
            TString res = Cin.ReadLine();

            return res;
        } catch (const TTerminalAttrs::TTermiosEx& e) {
            ::NTvmknife::NOutput::Out() << "Failed to hide secret in console with termios: " << e.what() << Endl;
            return Cin.ReadLine();
        }
    }
#else
    TString TConfidentialValue::ReadFromStdinImpl() {
        return Cin.ReadLine();
    }
#endif

    void TModeParse::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts, 1);

        opts.SetTitle("This mode can parse any type of ticket");

        opts.SetExamples("tvmknife parse_ticket 3:serv:CK0VEML2mNQFIgYI8gEQ8gE:THEjBMIEs9_...");

        opts.AddLongOption('u', "unittest", "use public keys from 'tvmknife unittest public_keys'")
            .StoreTrue(&Unittest_)
            .NoArgument();

        opts.AddLongOption('t', "ticket", "Obsolete and noop - will be deleted later")
            .NoArgument()
            .Hidden();
    }

    int TModeParse::DoRun(NLastGetopt::TOptsParseResult&& parsedOptions) {
        const TString ticket = parsedOptions.GetFreeArgs().at(0);

        try {
            if (Unittest_) {
                Cout << ParseAnyTicketUnittest(ticket);
            } else {
                Cout << ParseAnyTicket(ticket);
            }
        } catch (...) {
            Cerr << "Ticket is malformed" << Endl;
            throw;
        }

        return 0;
    }

    void TModeParseKeys::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts, 1);

        opts.SetTitle("This mode can parse tvm keys");

        opts.SetExamples("tvmknife parse_keys 1:CK0VEML2mNQFIgYI8gEQ8gE:THEjBMIEs9_...");
    }

    int TModeParseKeys::DoRun(NLastGetopt::TOptsParseResult&& parsedOptions) {
        const TString keys = parsedOptions.GetFreeArgs().at(0);
        try {
            Cout << ParseKeys(keys);
        } catch (...) {
            Cerr << "Keys are malformed" << Endl;
            throw;
        }

        return 0;
    }

    void TModeServiceTicket::TModeClCred::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);

        opts.SetTitle("This mode can get service ticket from tvm-api.yandex.net\n");

        opts.AddSection("Example",
                        "echo 'xxxx6QB7tNC37dY_EL55Fg' | tvmknife get_service_ticket client_credentials -s 123 -d 117");
        opts.AddSection("Example",
                        "tvmknife get_service_ticket client_credentials -s 123 -d 117\n"
                        "Enter 'tvm_secret':");
        opts.AddSection("Example",
                        "tvmknife get_service_ticket client_credentials -s 123 -d 117 -S 'path/to/secret'\n");

        opts.AddLongOption('s', "src", "source of request with service-ticket")
            .StoreResult(&Src_)
            .Required();
        opts.AddLongOption('d', "dst", "destination of request with service-ticket")
            .StoreResult(&Dst_)
            .Required();

        Secret_ = std::make_unique<TConfidentialValue>(opts, 'S', "secret", "secret of 'src'");
    }

    int TModeServiceTicket::TModeClCred::DoRun(NLastGetopt::TOptsParseResult&&) {
        Cout << TTvmProxy::ClientCred(Src_, Dst_, Secret_->Get());
        return 0;
    }

    void TModeServiceTicket::TModeSshKey::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);

        opts.SetTitle("This mode can get service ticket for debug purpose.\n"
                      "Public part of ssh key is expected to be published to Staff\n"
                      "User requires the role 'TVM ssh user' in ABC in service associated with 'src'.\n"
                      "Tvmknife uses ssh-agent by default, so you need to ensure it is operating properly"
                      " or you need specify private key (-k).");

        opts.SetExamples("tvmknife get_ticket sshkey -s 123 -d 456");

        opts.AddLongOption('s', "src", "source of request with service-ticket")
            .StoreResult(&CliSrc_)
            .Required();
        opts.AddLongOption('d', "dst", "destination of request with service-ticket")
            .StoreResult(&CliDst_)
            .Required();
        opts.AddLongOption('l', "login", "from Staff")
            .StoreResult(&Login_)
            .DefaultValue("");
        opts.AddLongOption('k', "key", "private part of ssh key.\n"
                                       "If empty, try to use ssh-agent")
            .StoreResult(&Key_)
            .DefaultValue("");
    }

    int TModeServiceTicket::TModeSshKey::DoRun(NLastGetopt::TOptsParseResult&&) {
        TAliases aliases;
        ui32 src = 0;
        ui32 dst = 0;

        if (!TryIntFromString<10>(CliSrc_, src)) {
            auto aliased_src = aliases.ResolveAlias("get_service_ticket", CliSrc_);
            if (aliased_src) {
                src = *aliased_src;
            } else {
                Cerr << "No alias found for src " << CliSrc_ << Endl;
                return 1;
            }
        }

        if (!TryIntFromString<10>(CliDst_, dst)) {
            auto aliased_dst = aliases.ResolveAlias("get_service_ticket", CliDst_);
            if (aliased_dst) {
                dst = *aliased_dst;
            } else {
                Cerr << "No alias found for dst " << CliDst_ << Endl;
                return 1;
            }
        }

        Cout << TTvmProxy::Sshkey(src, dst, Login_, Key_);

        return 0;
    }

    TModeServiceTicket::TModeServiceTicket()
        : ModeClCred_(std::make_unique<TModeClCred>())
        , ModeSshkey_(std::make_unique<TModeSshKey>())
    {
    }

    void TModeServiceTicket::RegisterModes(TModChooser& modes) {
        modes.DisableSvnRevisionOption();
        modes.SetDescription("This mode can get service ticket from tvm-api.yandex.net\n");

        modes.AddMode("client_credentials", ModeClCred_.get(), "use client credentials (secret)", false, false);
        modes.AddMode("sshkey", ModeSshkey_.get(), "use your ssh key", false, false);
    }

    TModeUserTicket::TCommonOpts::TCommonOpts(NLastGetopt::TOpts& opts, bool addBlackbox) {
        if (addBlackbox) {
            opts.AddLongOption('b', "blackbox", "destination of request with service-ticket")
                .StoreResult(&Blackbox)
                .DefaultValue("https://blackbox-test.yandex.net")
                .Completer(Choises(NBbProxy::SuggestBlackboxes()));
        }

        opts.AddLongOption('i', "tvm_id", "your TVM client (src)").StoreResult(&Src).Required();

        TvmSecret = std::make_unique<TConfidentialValue>(
            opts, 'S', "tvm_secret", "secret of 'src'.\nUse grant_type=client_credentials instead of grant_type=sshkey to get service ticket");

        opts.AddLongOption('L', "ssh_login", "specify login to use your sshkey for grant_type=sshkey")
            .StoreResult(&Login);
    }

    void TModeUserTicket::TCommonOpts::FixBlackbox(const TString& override) {
        if (override) {
            Blackbox = override;
        }
        if (!Blackbox.StartsWith("http")) {
            Blackbox = "https://" + Blackbox;
        }
        ::NTvmknife::NOutput::Out() << "Using blackbox: " << Blackbox << Endl;
    }

    TString TModeUserTicket::TCommonOpts::GetServiceTicket() const {
        if (!TvmSecret->IsDefault() && Login) {
            ythrow yexception() << "options 'tvm_secret' and 'ssh_login' conflicts with each other: "
                                   "they describes different ways for getting service ticket";
        }
        const ui32 dst = IntFromString<ui32, 10>(NBbProxy::BbTvmId(Blackbox));

        if (!TvmSecret->IsDefault()) {
            ::NTvmknife::NOutput::Out() << "Using grant_type=client_credentials to get service ticket" << Endl;
            return TTvmProxy::ClientCred(Src, dst, TvmSecret->Get());
        }
        ::NTvmknife::NOutput::Out() << "Using grant_type=sshkey to get service ticket" << Endl;
        return TTvmProxy::Sshkey(Src, dst, Login, "");
    }

    TString TModeUserTicket::TCommonOpts::GetUserTicket(const NBbProxy::TCredential& cred) const {
        TString ut = cred.TryGetFromCache();
        return ut ? ut : NBbProxy::GetUserTicket(cred, GetServiceTicket());
    }

    TString TModeUserTicket::TCommonOpts::GetLogin() const {
        return Login ? Login : TSshUtils::GetLogin();
    }

    void TModeUserTicket::TModeOAuth::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);
        CommonOpts_ = std::make_unique<TCommonOpts>(opts);

        opts.SetTitle(
            "This mode will get user ticket with 2 steps:\n"
            "  1) get service-ticket (with sshkey by default) - from tvm-api\n"
            "  2) use oauth token and service-ticket to get user-ticket - from blackbox\n"
            "\n"
            "Ssh key will be used via ssh-agent with fallback to your key in ~/.ssh/\n"
            "\n"
            "OAuth token and service-ticket are required to provide secure enough access to blackbox: https://st.yandex-team.ru/PASSP-30722.\n"
            "OAuth token provides user's will. Service-ticket is required for accounting in blackbox.");

        opts.AddSection("Requirements",
                        "You need role 'TVM ssh user' in ABC-service of your tvmid to use sshkey (for step 1).\n"
                        "https://wiki.yandex-team.ru/passport/tvm2/getabcrole/\n"
                        "You can get ABC-service by tvmid: `tvmknife info ...`");
        opts.AddSection("Requirements",
                        "OAuth token must be xtoken to add scopes 'bb:sess...'");

        opts.AddSection("Example",
                        "echo '6QB7tNC37dY_EL55Fg...' | tvmknife get_user_ticket oauth -b 'blackbox.yandex-team.net' --tvm_id=100500");
        opts.AddSection("Example",
                        "tvmknife get_user_ticket oauth -b 'blackbox.yandex-team.net' --tvm_id=123 -S\n"
                        "Enter 'token':\n"
                        "Enter 'tvm_secret':");
        opts.AddSection("Example",
                        "echo '6QB7tNC37dY_EL55Fg...' | tvmknife get_user_ticket oauth -b 'blackbox.yandex-team.net' --tvm_id=100500 -S 'path/to/secret");

        opts.AddSection("Fast way to create OAuth token (will be valid for 24 hours only)",
                        TOAuth::ListLinks());

        Token_ = std::make_unique<TConfidentialValue>(opts, 't', "token", "your OAuth token");

        opts.AddLongOption("add_scope_sessionid",
                           "add scope 'bb:sessionid' to user-ticket (oauth token must be xtoken)")
            .StoreTrue(&AddScopeSessionid_)
            .NoArgument();
        opts.AddLongOption("add_scope_sessguard",
                           "add scope 'bb:sessguard' to user-ticket (oauth token must be xtoken)")
            .StoreTrue(&AddScopeSessguard_)
            .NoArgument();
    }

    int TModeUserTicket::TModeOAuth::DoRun(NLastGetopt::TOptsParseResult&&) {
        TBbClient::TDebugUTParams params{
            .OauthToken = Token_->Get(),
            .AddScopeSessionid = AddScopeSessionid_,
            .AddScopeSessguard = AddScopeSessguard_,
        };

        CommonOpts_->FixBlackbox();
        NBbProxy::TOAuthCredential cred(CommonOpts_->Blackbox, params);

        Cout << CommonOpts_->GetUserTicket(cred);

        return 0;
    }

    void TModeUserTicket::TModeSshkey::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);
        CommonOpts_ = std::make_unique<TCommonOpts>(opts, false);

        opts.SetTitle(
            "This mode will get user ticket with 3 steps:\n"
            "  1) use your ssh key to get oauth token - from oauth\n"
            "  2) use your ssh key to get service-ticket - from tvm-api\n"
            "  3) use oauth token and service-ticket to get user-ticket - from blackbox\n"
            "\n"
            "Ssh key will be used via ssh-agent with fallback to your key in ~/.ssh/\n"
            "This mode works only with blackbox.yandex-team.ru, because all ssh keys belong to staff.yandex-team.ru.\n"
            "'oauth_client_id' also must be from oauth.yandex-team.ru\n"
            "\n"
            "OAuth token and service-ticket are required to provide secure enough access to blackbox: https://st.yandex-team.ru/PASSP-30722.\n"
            "OAuth token provides user's will. Service-ticket is required for accounting in blackbox.");

        opts.AddSection("Requirements",
                        "You need role 'TVM ssh user' in ABC-service of your tvmid to use sshkey (for step 2).\n"
                        "https://wiki.yandex-team.ru/passport/tvm2/getabcrole/\n"
                        "You can get ABC-service by tvmid: `tvmknife info ...`");
        opts.AddSection("Requirements",
                        "OAuth token must be xtoken to add scopes 'bb:sess...'");

        opts.AddSection("Example",
                        "tvmknife get_user_ticket sshkey --tvm_id=100500");

        opts.AddLongOption("add_scope_sessionid",
                           "add scope 'bb:sessionid' to user-ticket (oauth token must be xtoken)")
            .StoreTrue(&AddScopeSessionid_)
            .NoArgument();
        opts.AddLongOption("add_scope_sessguard",
                           "add scope 'bb:sessguard' to user-ticket (oauth token must be xtoken)")
            .StoreTrue(&AddScopeSessguard_)
            .NoArgument();

        opts.AddLongOption("oauth_client_id",
                           "use specified client_id to create oauth token")
            .StoreResult(&OauthClientId_)
            .DefaultValue(TOAuth::DEFAULT_YATEAM_CLIENT_ID);
        opts.AddLongOption("oauth_client_secret",
                           "use specified client_secret to create oauth token")
            .StoreResult(&OauthClientSecret_);
    }

    int TModeUserTicket::TModeSshkey::DoRun(NLastGetopt::TOptsParseResult&&) {
        if (OauthClientSecret_.empty()) {
            if (OauthClientId_ == TOAuth::DEFAULT_YATEAM_CLIENT_ID) {
                OauthClientSecret_ = TOAuth::DEFAULT_YATEAM_CLIENT_SECRET;
            } else {
                ythrow yexception() << "missing 'oauth_client_secret'";
            }
        }

        const TString oauthToken = TOAuth::GetToken(
            TOAuth::TParams{
                OauthClientId_,
                OauthClientSecret_,
                CommonOpts_->GetLogin(),
            });
        // Do not cache oauthToken on disk: PASSP-30722

        TBbClient::TDebugUTParams params{
            .OauthToken = oauthToken,
            .AddScopeSessionid = AddScopeSessionid_,
            .AddScopeSessguard = AddScopeSessguard_,
        };

        CommonOpts_->FixBlackbox("https://blackbox.yandex-team.ru");
        NBbProxy::TOAuthCredential cred(CommonOpts_->Blackbox, params);

        Cout << CommonOpts_->GetUserTicket(cred);

        return 0;
    }

    int TModeUserTicket::TModeListBlackboxes::DoRun(NLastGetopt::TOptsParseResult&&) {
        Cout << NPassport::NTvmknife::NBbProxy::ListBlackboxes();
        return 0;
    }

    TModeUserTicket::TModeUserTicket()
        : ModeOAuth_(std::make_unique<TModeOAuth>())
        , ModeSshkey_(std::make_unique<TModeSshkey>())
        , ModeListBlackboxes_(std::make_unique<TModeListBlackboxes>())
    {
    }

    void TModeUserTicket::RegisterModes(TModChooser& modes) {
        modes.DisableSvnRevisionOption();
        modes.SetDescription("This mode can get user ticket from blackbox\n"
                             "NB: you need network access to blackbox and grants for your tvm_id");

        modes.AddMode("oauth", ModeOAuth_.get(), "use oauth token to get user ticket (without grants)", false, false);
        modes.AddMode("sshkey", ModeSshkey_.get(), "use sshkey to get user ticket (without grants)", false, false);
        modes.AddMode("list_blackboxes", ModeListBlackboxes_.get(), "only list all blackboxes", false, false);
    }

    void TModeUnittest::TModeService::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);

        opts.SetTitle("This mode can generate service-ticket for unittests");
        opts.SetExamples("tvmknife unittest service -s 123 -d 117");
        opts.AddLongOption('s', "src", "source of request with service-ticket")
            .StoreResult(&Src_)
            .Required();
        opts.AddLongOption('d', "dst", "destination of request with service-ticket")
            .StoreResult(&Dst_)
            .Required();
        opts.AddLongOption("scopes", "list of scopes.\n"
                                     "Example: bb:pwd,bb:sess")
            .StoreResult(&Scopes_)
            .DefaultValue("");
        opts.AddLongOption('i', "issuer_uid", "uid of developer who got service-ticket with grant_type=sshkey")
            .StoreResult(&IssuerUid_);
    }

    int TModeUnittest::TModeService::DoRun(NLastGetopt::TOptsParseResult&& parsedOptions) {
        if (Src_ == 0) {
            ythrow yexception() << "'src' should be positive";
        }
        if (Dst_ == 0) {
            ythrow yexception() << "'dst' should be positive";
        }
        if (parsedOptions.Has('i') && IssuerUid_ == 0) {
            ythrow yexception() << "'issuer uid' should be unsigned int";
        }

        std::vector<TString> scopesArr = NPassport::NUtils::ToVector<TString>(Scopes_, ',');

        Cout << TUnittest::GetServiceTicket(Src_, Dst_, scopesArr, IssuerUid_);

        return 0;
    }

    void TModeUnittest::TModeUser::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts);

        opts.SetTitle("This mode can generate user-ticket for unittests");
        opts.SetExamples("tvmknife unittest user -d 100500");

        opts.AddLongOption('u', "uids", "list of uids in user ticket. Example: 123,456")
            .StoreResult(&UidsStr_);

        opts.AddLongOption('d', "default", "default uid in user ticket - may be 0. Example: 789")
            .StoreResult(&DefaultUid_)
            .Required();

        opts.AddLongOption('e', "env", "passport environment.\n"
                                       "Supported: prod, test, prod_yateam, test_yateam, stress")
            .StoreResult(&Env_)
            .DefaultValue("test")
            .Completer(Choises({"prod", "test", "prod_yateam", "test_yateam", "stress"}));

        opts.AddLongOption("scopes", "list of scopes. Example: bb:pwd,bb:sess")
            .StoreResult(&Scopes_)
            .DefaultValue("");
    }

    int TModeUnittest::TModeUser::DoRun(NLastGetopt::TOptsParseResult&&) {
        std::vector<ui64> uids;
        TStringBuf u(UidsStr_);
        while (u) {
            TStringBuf uid = u.NextTok(',');
            ui64 num = 0;
            Y_ENSURE(TryIntFromString<10>(uid, num) && num > 0, "Uid should be positive ui64: " << uid);
            uids.push_back(num);
        }

        std::vector<TString> scopesArr = NPassport::NUtils::ToVector<TString>(Scopes_, ',');

        Cout << TUnittest::GetUserTicket(DefaultUid_, uids, scopesArr, Env_);

        return 0;
    }

    int TModeUnittest::TModeKeys::DoRun(NLastGetopt::TOptsParseResult&&) {
        Cout << TUnittest::GetPublicKeys();
        return 0;
    }

    TModeUnittest::TModeUnittest()
        : ModeService_(std::make_unique<TModeService>())
        , ModeUser_(std::make_unique<TModeUser>())
        , ModeKeys_(std::make_unique<TModeKeys>())
    {
    }

    void TModeUnittest::RegisterModes(TModChooser& modes) {
        modes.DisableSvnRevisionOption();
        modes.SetDescription("This mode can generate ticket for unittests\n"
                             "This ticket can be checked only with public keys from mode 'public_keys'\n");

        modes.AddMode("service", ModeService_.get(), "generate service-ticket", false, false);
        modes.AddMode("user", ModeUser_.get(), "generate user-ticket", false, false);
        modes.AddMode("public_keys", ModeKeys_.get(), "get public keys to check tickets generated by unittest mode", false, false);
    }

    void TModeInfo::RegisterOptions(NLastGetopt::TOpts& opts) {
        AddDefaultOpts(opts, 1);

        opts.SetTitle("This mode can get some info about client and service");
        opts.SetExamples("tvmknife info 222");

        opts.AddLongOption("add_secret_link", "adds link to tvm secret in yav")
            .NoArgument()
            .StoreTrue(&AddSecretLink_);

        opts.AddLongOption("add_creator", "adds creator of tvm client")
            .NoArgument()
            .StoreTrue(&AddCreator_);

        opts.AddLongOption('i', "tvm_id", "tvm client_id. Obsolete and noop - will be deleted later")
            .NoArgument()
            .Hidden();
    }

    int TModeInfo::DoRun(NLastGetopt::TOptsParseResult&& parsedOptions) {
        const TString arg = parsedOptions.GetFreeArgs().at(0);

        NTvmAuth::TTvmId id = 0;
        if (!TryIntFromString<10>(arg, id) || id == 0) {
            ythrow yexception() << "'src' should be unsigned int";
        }

        Cout << TInfo::GetInfo(arg, TInfo::TAdd{
                                        .Yav = AddSecretLink_,
                                        .Creator = AddCreator_,
                                    });

        return 0;
    }

    int Run(int argc, const char** argv) {
        TModChooser chooser;
        chooser.SetDescription("Tool for debugging and testing with TVM tickets\n"
                               "Site: https://wiki.yandex-team.ru/passport/tvm2\n");

        auto p = MakeHolder<TModeParse>();
        auto k = MakeHolder<TModeParseKeys>();
        auto t = MakeHolder<TModeServiceTicket>();
        auto b = MakeHolder<TModeUserTicket>();
        auto u = MakeHolder<TModeUnittest>();
        auto i = MakeHolder<TModeInfo>();

        chooser.AddMode("parse_ticket", p.Get(), "parse any ticket", false, false);
        chooser.AddMode("parse_keys", k.Get(), "parse keys from TVM-API", false, true);
        chooser.AddMode("get_service_ticket", t.Get(), "get ticket from TVM-API", false, false);
        chooser.AddMode("get_user_ticket", b.Get(), "get ticket from blackbox", false, false);
        chooser.AddMode("unittest", u.Get(), "generate permanent ticket for unittest", false, false);
        chooser.AddMode("info", i.Get(), "get info about TVM client", false, false);

        chooser.AddCompletions("tvmknife");

        return chooser.Run(argc, argv);
    }
}
