#include "tvm_client.h"

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/tvmauth/version.h>
#include <library/cpp/tvmauth/deprecated/service_context.h>
#include <library/cpp/tvmknife/output.h>
#include <library/cpp/tvmknife/simple_tvm_client.h>

#include <util/generic/yexception.h>
#include <util/string/cast.h>

namespace {
    class TClientUtils: NTvmknife::TSimpleTvmClient {
    public:
        using TSimpleTvmClient::PrettifyJson;
    };
}

namespace NPassport::NTvmknife {
    TString TTvmClient::PublicKeys() {
        const TString url = TString("/2/keys?lib_version=tvmknife_") + NTvmAuth::LibVersion();

        TString out;
        try {
            TStringOutput s(out);
            TSimpleHttpClient http("https://tvm-api.yandex.net", 443);
            http.DoGet(url, &s);
        } catch (const std::exception& e) {
            ::NTvmknife::NOutput::Out() << "Failed to fetch public keys from TVM:" << Endl
                                        << e.what()
                                        << TClientUtils::PrettifyJson(out) << Endl;
        }

        return out;
    }

    TString TTvmClient::ClientCredentials(const TString& src,
                                          const TString& dst,
                                          TStringBuf secret) {
        const TString ts = IntToString<10>(time(nullptr));

        TString post;
        post.append("grant_type=client_credentials");
        post.append("&ts=").append(ts);
        post.append("&src=").append(src);
        post.append("&dst=").append(dst);
        post.append("&tvmknife_").append(NTvmAuth::LibVersion());

        auto ctx = NTvmAuth::TServiceContext::SigningFactory(secret);
        post.append("&sign=").append(ctx.SignCgiParamsForTvm(ts, dst));

        TString out;
        try {
            TStringOutput s(out);
            TSimpleHttpClient http("https://tvm-api.yandex.net", 443);
            http.DoPost("/2/ticket", post, &s);
        } catch (const std::exception& e) {
            ythrow yexception() << "Failed to fetch ticket from TVM:" << Endl
                                << e.what()
                                << TClientUtils::PrettifyJson(out) << Endl;
        }

        return out;
    }

    TString TTvmClient::Info(const TString& id) {
        TString out;
        try {
            TStringOutput s(out);
            TSimpleHttpClient http("https://tvm.yandex-team.ru", 443);
            http.DoGet("/client/" + id + "/info?with_deleted=true", &s);
        } catch (const THttpRequestException& e) {
            if (e.GetStatusCode() == 404) {
                return {};
            }
            ythrow yexception() << "Failed to fetch info from TVM:" << Endl
                                << e.what()
                                << TClientUtils::PrettifyJson(out) << Endl;
        } catch (const std::exception& e) {
            ythrow yexception() << "Failed to fetch info from TVM:" << Endl
                                << e.what()
                                << TClientUtils::PrettifyJson(out) << Endl;
        }

        return out;
    }
}
