#include "common.h"

#include <passport/infra/tools/tvmknife/src/blackbox_proxy.h>
#include <passport/infra/tools/tvmknife/src/cache.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

using namespace NPassport::NTvmknife;

Y_UNIT_TEST_SUITE(BbProxy) {
    Y_UNIT_TEST(tvmId) {
        UNIT_ASSERT_STRINGS_EQUAL("222", NBbProxy::BbTvmId("https://blackbox.yandex.net/"));
        UNIT_ASSERT_STRINGS_EQUAL("222", NBbProxy::BbTvmId("http://blackbox-rc.yandex.net/"));
        UNIT_ASSERT_STRINGS_EQUAL("222", NBbProxy::BbTvmId("http://blackbox-mail.yandex.net"));

        UNIT_ASSERT_STRINGS_EQUAL("224", NBbProxy::BbTvmId("blackbox-test.yandex.net/"));
        UNIT_ASSERT_STRINGS_EQUAL("224", NBbProxy::BbTvmId("pass-test.yandex.ru"));

        UNIT_ASSERT_STRINGS_EQUAL("223", NBbProxy::BbTvmId("blackbox.yandex-team.ru"));

        UNIT_ASSERT_STRINGS_EQUAL("225", NBbProxy::BbTvmId("blackbox-test.yandex-team.ru"));

        UNIT_ASSERT_STRINGS_EQUAL("226", NBbProxy::BbTvmId("blackbox-stress.yandex.net"));
        UNIT_ASSERT_STRINGS_EQUAL("226", NBbProxy::BbTvmId("pass-stress-f1.sezam.yandex.net"));
        UNIT_ASSERT_STRINGS_EQUAL("226", NBbProxy::BbTvmId("pass-stress-i1.sezam.yandex.net"));
        UNIT_ASSERT_STRINGS_EQUAL("226", NBbProxy::BbTvmId("pass-stress-m1.sezam.yandex.net"));
        UNIT_ASSERT_STRINGS_EQUAL("226", NBbProxy::BbTvmId("pass-stress-s1.sezam.yandex.net"));

        UNIT_ASSERT_STRINGS_EQUAL("239", NBbProxy::BbTvmId("blackbox-mimino.yandex.net"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(NBbProxy::BbTvmId("kek"),
                                       yexception,
                                       R"(Unknown blackbox host 'kek'.
Please choose one of:
blackbox.yandex.net -> tvm_id=222
blackbox-rc.yandex.net -> tvm_id=222
blackbox-mail.yandex.net -> tvm_id=222
blackbox-test.yandex.net -> tvm_id=224
pass-test.yandex.ru -> tvm_id=224
blackbox.yandex-team.ru -> tvm_id=223
blackbox-test.yandex-team.ru -> tvm_id=225
blackbox-stress.yandex.net -> tvm_id=226
pass-stress-f1.sezam.yandex.net -> tvm_id=226
pass-stress-i1.sezam.yandex.net -> tvm_id=226
pass-stress-m1.sezam.yandex.net -> tvm_id=226
pass-stress-s1.sezam.yandex.net -> tvm_id=226
blackbox-mimino.yandex.net -> tvm_id=239
)");
    }

    Y_UNIT_TEST(blackboxes) {
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net -> tvm_id=222\n"
                                 "blackbox-rc.yandex.net -> tvm_id=222\n"
                                 "blackbox-mail.yandex.net -> tvm_id=222\n"
                                 "blackbox-test.yandex.net -> tvm_id=224\n"
                                 "pass-test.yandex.ru -> tvm_id=224\n"
                                 "blackbox.yandex-team.ru -> tvm_id=223\n"
                                 "blackbox-test.yandex-team.ru -> tvm_id=225\n"
                                 "blackbox-stress.yandex.net -> tvm_id=226\n"
                                 "pass-stress-f1.sezam.yandex.net -> tvm_id=226\n"
                                 "pass-stress-i1.sezam.yandex.net -> tvm_id=226\n"
                                 "pass-stress-m1.sezam.yandex.net -> tvm_id=226\n"
                                 "pass-stress-s1.sezam.yandex.net -> tvm_id=226\n"
                                 "blackbox-mimino.yandex.net -> tvm_id=239\n",
                                 NBbProxy::ListBlackboxes());
    }

    Y_UNIT_TEST(getOauth) {
        TPortManager pm;
        ui16 port = pm.GetPort(80);
        NMock::TMockServer server(port, []() { return new NMock::TBlackbox; });

        const TString blackbox = TStringBuilder() << "localhost:" << port;
        NBbProxy::AddBbTvmId("localhost");
        NFs::RemoveRecursive("./cache/");
        TCache::Get().SetCacheDir("./cache/");

        NBbProxy::TOAuthCredential cred1(blackbox,
                                         TBbClient::TDebugUTParams{"token_1"});
        NBbProxy::TOAuthCredential cred2(blackbox,
                                         TBbClient::TDebugUTParams{"token_2"});
        UNIT_ASSERT_VALUES_EQUAL("", cred1.TryGetFromCache());
        UNIT_ASSERT_VALUES_EQUAL("", cred2.TryGetFromCache());

        UNIT_ASSERT_VALUES_EQUAL("user_ticket_1\n",
                                 NBbProxy::GetUserTicket(cred1, "service_ticket"));

        UNIT_ASSERT_VALUES_EQUAL("user_ticket_1\n", cred1.TryGetFromCache());
        UNIT_ASSERT_VALUES_EQUAL("", cred2.TryGetFromCache());

        UNIT_ASSERT_VALUES_EQUAL("user_ticket_2\n",
                                 NBbProxy::GetUserTicket(cred2, "service_ticket"));

        UNIT_ASSERT_VALUES_EQUAL("user_ticket_1\n", cred1.TryGetFromCache());
        UNIT_ASSERT_VALUES_EQUAL("user_ticket_2\n", cred2.TryGetFromCache());
    }
}
