#include "check_processor.h"

#include "connection.h"
#include "func.h"
#include "scenario.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/rapidjson/include/rapidjson/document.h>
#include <contrib/libs/rapidjson/include/rapidjson/prettywriter.h>

#include <util/string/ascii.h>
#include <util/string/split.h>

#include <regex>

namespace NPassport::NLast {
    static const std::regex REGEX_HEADER("^([^\\(\\)<>@,;:\\\"/\\?=\\{\\} \t\\[]+):([[:space:]]+(.*))?\r\n");
    static const std::regex REGEX_STATUS_LINE("HTTP/[[:digit:]]\\.[[:digit:]] ([[:digit:]]+) .*\r\n");

    TCheckProcessor::TCheckProcessor(const TTestContext& ctx, const TPerformResult& result)
        : Ctx_(ctx)
        , PerformResult_(result)
        , CurResult_(Ctx_.Check->GetResult())
        , ExpectedCookies_(CurResult_->GetCookies())
        , ExpectedHeaders_(CurResult_->GetHeaders())
    {
    }

    bool TCheckProcessor::Process(TString& output) {
        bool res = false;
        try {
            res = CheckPostPerform(output);
            if (!Ctx_.ErrMsg.empty()) {
                ErrorMsg_ = Ctx_.ErrMsg;
                res = false;
            }
        } catch (const NCurlwrap::TCurlError& e) {
            ErrorMsg_ = "Curl error: ";
            ErrorMsg_.append(e.what());
            res = false;
        } catch (const std::exception& e) {
            ErrorMsg_ = e.what();
            res = false;
        }

        output = MakeOutput(output, res);
        return res;
    }

    TString TCheckProcessor::MakeOutput(const TString& output, bool isSuccess) const {
        TStringStream ss;
        if (!isSuccess) {
            bool json = ContentType_.Contains("json");

            ss << output;

            ss << Endl
               << PerformResult_.Output;
            if (Ctx_.IsPost) {
                ss << PerformResult_.Request << Endl
                   << Endl;
            }
            ss << PerformResult_.InHdr << Endl;
            if (json) {
                rapidjson::Document d;
                if (d.Parse(PerformResult_.Page.c_str()).HasParseError()) {
                    ss << "Fatal std::exception: couldn't parse json reponse" << Endl;
                    ss << PerformResult_.Page << Endl;
                }

                rapidjson::StringBuffer s;
                rapidjson::PrettyWriter<rapidjson::StringBuffer> wr(s);
                d.Accept(wr);
                ss << s.GetString() << Endl;
            } else {
                ss << PerformResult_.Page << Endl;
            }

            ss << "*** Error (combination #" << Ctx_.Idx << "): '" << ErrorMsg_ << "'" << Endl;
            ss << Endl
               << "Current variables are:" << Endl;
            for (const auto& v : Ctx_.Vars) {
                if (v) {
                    v->Print(ss);
                }
            }
            ss << Endl
               << "Current check is:" << Endl;
            Ctx_.Check->Print(ss);
        }

        // Shall we print out HTTP trace?
        if (TConfig::Get().ShowHttpTrace && isSuccess) {
            ss << Endl
               << PerformResult_.Output;
            if (Ctx_.IsPost) {
                ss << PerformResult_.Request << Endl
                   << Endl;
            }
            ss << PerformResult_.InHdr << Endl;
            ss << PerformResult_.Page;
        }

        return ss.Str();
    }

    bool TCheckProcessor::CheckHeaderLine(TStringBuf header) {
        if (header == "\r\n") {
            return true;
        }

        // All lines following the Status Line are headers. Parse
        // every line into header-name and header-value.
        //
        // Process Set-Cookie header only.
        // BUGBUG!!! Hedaer name is a 'token' so it cannot contain ']'
        // as well, but we cannot include this in our regex lacks --
        // it stops working. Bug in the regex lib?
        std::smatch groups;
        if (!std::regex_match(header.begin(), header.end(), groups, REGEX_HEADER)) {
            TStringStream msg;
            msg << "Couldn't parse HTTP header line. The line follows:" << Endl
                << header;
            ErrorMsg_ = msg.Str();
            return false;
        }

        TString name(groups.str(1));
        TString value;
        if (groups.size() > 2) {
            value = groups.str(3);
        }

        if (AsciiEqualsIgnoreCase(name, "Content-type")) {
            ContentType_ = value;
        }

        if (AsciiEqualsIgnoreCase(name, "Set-Cookie")) {
            if (!CheckCookie(value)) {
                return false;
            }
        } else {
            auto it = ExpectedHeaders_.find(name);
            if (it != ExpectedHeaders_.end()) {
                bool res = it->second->Match(Ctx_, value, false);
                ExpectedHeaders_.erase(it);
                if (!res) {
                    TStringStream s;
                    s << "Header: \"" << name << "\" has unexpected value: \"" << value << "\"";
                    ErrorMsg_ = s.Str();
                }
                return res;
            }
        }

        return true;
    }

    bool TCheckProcessor::CheckStatusLine(TStringBuf line) {
        // Parse Status Line
        std::smatch groups;

        if (!std::regex_match(line.begin(), line.end(), groups, REGEX_STATUS_LINE)) {
            TStringStream msg;
            msg << "Couldn't parse HTTP Status Line. The line follows:" << Endl;
            msg << line;
            ErrorMsg_ = msg.Str();
            return false;
        }

        // Check Status-Code
        long code = IntFromString<ui32, 10>(groups.str(1));
        if (code != CurResult_->GetStatusCode()) {
            TStringStream msg;
            msg << "Expected Status-Code " << CurResult_->GetStatusCode()
                << ", got the following:" << Endl;
            msg << line;
            ErrorMsg_ = msg.Str();
            return false;
        }

        return true;
    }

    bool TCheckProcessor::CheckCookie(const TString& cookie) {
        const std::regex regexCookie("^([^;=]+)[[:space:]]*=[[:space:]]*([^;=]*)"
                                     "(([[:space:]]*;[[:space:]]*([[:alpha:]-]+)[[:space:]]*(=[[:space:]]*([^;]+))?)*)"
                                     "([[:space:]]*;[[:space:]]*)?$");

        // Parse cookie string
        std::smatch groupsCookie;
        if (!std::regex_match(cookie.cbegin(), cookie.cend(), groupsCookie, regexCookie)) {
            TStringStream msg;
            msg << "Couldn't parse HTTP Set-Cookie string:" << Endl
                << cookie;
            ErrorMsg_ = msg.Str();
            return false;
        }

        // Assign name and value
        TString name(groupsCookie.str(1));
        TString value(groupsCookie.str(2));

        // Try find this cookie among expected ones
        auto it = std::find_if(ExpectedCookies_.begin(),
                               ExpectedCookies_.end(),
                               [&name](const std::shared_ptr<TResultCookie>& p) {
                                   return AsciiEqualsIgnoreCase(p->GetName(), name);
                               });
        if (it == ExpectedCookies_.end()) {
            TStringStream msg;
            msg << "Unexpected cookie:" << Endl
                << cookie;
            ErrorMsg_ = msg.Str();
            return false;
        }

        // If this cookie is not a don't-care type, perform all the required checks
        if (!(*it)->DontCare()) {
            // Check cookie value
            if (!(*it)->MatchValue(Ctx_, value)) {
                TStringStream msg;
                msg << "Expected Cookie value '" << (*it)->GetValue();
                msg << "', got:" << Endl
                    << cookie;
                ErrorMsg_ = msg.Str();
                return false;
            }

            // Check cookie attributes: compile attributes regexp which will cut off
            // of the tail one attribute at a time. Loop until no attributes found.
            // On every loop iteration look up the attribute among expected ones and
            // check its value. In the end check to see that all expected attributes
            // were found.
            TString attribStr(groupsCookie.str(3));
            const std::regex regexAttrib("[[:space:]]*;[[:space:]]*([[:alpha:]-]+)[[:space:]]*(=[[:space:]]*([^;]+))?");
            std::smatch groupsAttr;

            TResultCookie::TAttribArray expected((*it)->GetAttribs());
            while (std::regex_search(attribStr.cbegin(), attribStr.cend(), groupsAttr, regexAttrib)) {
                TString attrname(groupsAttr.str(1));
                TString attrval(groupsAttr.str(3));

                // Do we expect this cookie attribute?
                TResultCookie::TAttribArray::iterator att = expected.find(attrname);
                if (att == expected.end()) {
                    TStringStream msg;
                    msg << "Unexpected cookie attribute " << attrname << ":" << Endl;
                    msg << cookie;
                    ErrorMsg_ = msg.Str();
                    return false;
                }

                // Check attribute value
                if (!att->second->Match(Ctx_, NUtils::Urldecode(attrval).c_str())) {
                    TStringStream msg;
                    msg << "Expected cookie attribute value " << attrname << "=";
                    msg << att->second->GetValue() << ", got: " << Endl
                        << cookie;
                    ErrorMsg_ = msg.Str();
                    return false;
                }

                // Remove just checked attribute from the expected list
                expected.erase(att);

                // Cut off just checked attribute from the attribute string
                attribStr.erase(0, groupsAttr.position(0) + groupsAttr.length(0));
            }

            TStringStream msg;
            for (const auto& pair : expected) {
                msg << "Missing expected cookie attribute " << pair.first << Endl;
            }
            if (!expected.empty()) {
                msg << "Got cookie:" << Endl
                    << cookie;
                ErrorMsg_ = msg.Str();
                return false;
            }
        }

        // Remove just checked cookie from the list of expected ones.
        ExpectedCookies_.erase(it);

        return true;
    }

    bool TCheckProcessor::CheckHeaders() {
        TStringBuf headers(PerformResult_.InHdr);
        TStringBuf statusLine;
        GetNext(headers, '\n', statusLine);
        size_t sumSize = statusLine.size() + 1;
        if (sumSize > PerformResult_.InHdr.size()) {
            throw yexception() << "Something is very bad: failed to split headers from response";
        }

        if (!CheckStatusLine(TStringBuf(statusLine.data(), statusLine.size() + 1))) {
            return false;
        }

        while (headers) {
            TStringBuf headerLine;
            GetNext(headers, '\n', headerLine);
            sumSize += headerLine.size() + 1;
            if (sumSize > PerformResult_.InHdr.size()) {
                throw yexception() << "Something is very bad: failed to split headers from response";
            }

            if (!CheckHeaderLine(TStringBuf(headerLine.data(), headerLine.size() + 1))) {
                return false;
            }
        }

        for (const auto& pair : ExpectedHeaders_) {
            if (pair.second && !pair.second->Match(Ctx_, "dontmatch", false)) {
                ErrorMsg_ = "Header must not be absent: " + pair.first;
                return false;
            }
        }

        return true;
    }

    bool TCheckProcessor::CheckPostPerform(TString& output) {
        if (!CheckHeaders()) {
            return false;
        }

        // Does page body meets our expectations?
        if (CurResult_->GetXmlBody() != nullptr && PerformResult_.Page.empty()) {
            throw yexception() << "Missing expected page body";
        }
        if (CurResult_->GetXmlBody() == nullptr && !PerformResult_.Page.empty()) {
            throw yexception() << "Unexpected page body";
        }

        auto itFormat = Ctx_.Cgis.find("format");
        if (itFormat != Ctx_.Cgis.end() && itFormat->second == "json") {
            CheckResultItem(output, CurResult_->GetJsonBody());
        } else {
            CheckResultItem(output, CurResult_->GetXmlBody());
        }

        // Have we seen all expected don't-care-type cookies? If not, throw.
        bool first = true;
        for (const auto& cookie : ExpectedCookies_) {
            if (cookie->DontCare()) {
                continue;
            }
            if (!first) {
                ErrorMsg_.append(", ");
            } else {
                first = false;
                ErrorMsg_.assign("Missing expected Set-Cookie: \n");
            }
            ErrorMsg_.append(cookie->GetName());
        }
        if (!first) {
            throw yexception() << ErrorMsg_;
        }

        return true;
    }

    void TCheckProcessor::CheckResultItem(TString& output, const TResultItem* item) {
        if (item == nullptr || PerformResult_.Page.empty()) {
            return;
        }

        bool check = false;
        try {
            check = item->Match(Ctx_, PerformResult_.Page, true);
        } catch (const TMatchError& e) {
            output.append("--------------------------------------------------------------------------------\n");
            output.append("Result match error: ").append(e.what()).append("\n");
            output.append("--------------------------------------------------------------------------------\n");
        }

        if (!check) {
            ErrorMsg_.assign("Page body failed the check: ");
            ErrorMsg_.append(item->GetValue());
            throw yexception() << ErrorMsg_;
        }
    }

}
