#include "check_json_point.h"

#include "common.h"

#include <contrib/libs/rapidjson/include/rapidjson/document.h>
#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

namespace NPassport::NLast::NCheck {
    static const TString JSON_IS_ARRAY = "__json_is_array__";
    bool CheckJsonPath(const TTestContext& ctx, const TString& input, const NPassport::NLast::TFunctionArgs& args) {
        rapidjson::Document d;
        if (d.Parse(input.c_str()).HasParseError()) {
            throw TLastError() << "CheckJsonPath(): couldn't parse json";
        }

        if (args.find(JSON_IS_ARRAY) == args.end()) {
            if (!d.IsObject()) {
                throw TLastError() << "CheckJsonPath(): json input is not object";
            }
        } else {
            if (!d.IsArray()) {
                throw TLastError() << "CheckJsonPath(): json input is not array";
            }
        }

        for (const auto& pair : args) {
            const TString& jsonPoint = pair.first;
            const TString& result = pair.second;

            if (jsonPoint == JSON_IS_ARRAY) {
                continue;
            }

            bool dontMatch = result == "dontmatch";
            try {
                rapidjson::Value* jsonValue = rapidjson::Pointer(jsonPoint.c_str()).Get(d);

                if (nullptr == jsonValue) {
                    if (dontMatch) {
                        continue;
                    }
                    throw TLastError() << "CheckJsonPath(): json point does not exist: " << jsonPoint;
                }
                if (dontMatch) {
                    throw TLastError() << "CheckJsonPath(): json point exists but must not: " << jsonPoint;
                }

                TString value;
                if (pair.second.ResMod() == TArg::None) {
                    if (jsonValue->IsBool()) {
                        value.assign(jsonValue->GetBool() ? "true" : "false");
                    } else if (jsonValue->IsInt64()) {
                        value.assign(IntToString<10>(jsonValue->GetInt64()));
                    } else if (jsonValue->IsString()) {
                        value.assign(jsonValue->GetString());
                    } else if (jsonValue->IsNull()) {
                        value.assign("__null__");
                    }
                } else if (pair.second.ResMod() == TArg::ArraySize) {
                    if (!jsonValue->IsArray()) {
                        throw TLastError() << "CheckJsonPath(): json point expects array: "
                                           << jsonPoint << ". res_mod=array_size";
                    }
                    value.assign(IntToString<10>(jsonValue->Size()));
                } else if (pair.second.ResMod() == TArg::KeyCount) {
                    if (!jsonValue->IsObject()) {
                        throw TLastError() << "CheckJsonPath(): json point expects object: "
                                           << jsonPoint << ". res_mod=key_count";
                    }
                    value.assign(IntToString<10>(jsonValue->MemberCount()));
                }

                if (!pair.second.Match(value, ctx)) {
                    throw TMatchError() << "Jsonpoint: <" << jsonPoint
                                        << "> original: <" << result
                                        << ">, actual: <" << value
                                        << ">, check: <" << pair.second.CheckInfo() << ">";
                }
            } catch (const TMatchError&) {
                throw;
            } catch (const TLastError&) {
                throw;
            } catch (const std::exception& e) {
                throw TLastError() << "CheckJsonPath(): couldn't process json point: "
                                   << jsonPoint << ": " << e.what();
            }
        }

        return true;
    }
}
