#include "check_oauth_token.h"

#include "common.h"

#include <passport/infra/tools/ylast/src/context/oauth_context.h>

#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NLast::NCheck {
    bool CheckOAuthToken(
        const TTestContext& ctx,
        const TString& input,
        const NPassport::NLast::TFunctionArgs& args) {
        NPassport::NLast::TFunctionArgs::const_iterator it;

        it = args.find("status");
        if (it == args.end()) {
            throw TLastError() << "CheckOAuthToken(): no 'status' argument";
        }

        NAuth::TOAuthToken token = TOAuthContext::GetInstance().OauthParser->ParseToken(input);

        if (token.Status() == NAuth::TOAuthToken::NOT_INITED) {
            throw TLastError() << "CheckOAuthToken(): OAuthTokenParser not initialized!";
        }

        if ((it->second.Value() == "valid" && token.Status() != NAuth::TOAuthToken::VALID) ||
            (it->second.Value() == "expired" && token.Status() != NAuth::TOAuthToken::EXPIRED) ||
            (it->second.Value() == "broken" && token.Status() != NAuth::TOAuthToken::SIGN_BROKEN) ||
            (it->second.Value() == "glogout" && token.Status() != NAuth::TOAuthToken::GLOGOUTED)) {
            CheckField("OAuth token", input, "status", it->second.Value(), IntToString<10>((int)token.Status()));
        }

        CheckStringField("OAuth token", ctx, input, args, "version", IntToString<10>(token.Version()));
        CheckStringField("OAuth token", ctx, input, args, "uid", token.Uid() ? token.Uid() : "0");

        it = args.find("create_time");
        if (it != args.end()) {
            TString strval = SubstituteVars(it->second.Value(), ctx);
            time_t val = NUtils::ToUInt(strval, "create_time");
            if (!TimesClose(val, token.CreateTime())) {
                // use "checkField" to fail the check and report error
                CheckField("OAuth token", input, "create_time", strval, IntToString<10>(token.CreateTime()));
            }
        }

        it = args.find("expire_time");
        if (it != args.end()) {
            TString strval = SubstituteVars(it->second.Value(), ctx);
            time_t val = NUtils::ToUInt(strval, "expire_time");
            if (!TimesClose(val, token.ExpireTime())) {
                // use "checkField" to fail the check and report error)
                CheckField("OAuth token", input, "expire_time", strval, IntToString<10>(token.ExpireTime()));
            }
        }

        CheckStringField("OAuth token", ctx, input, args, "client_id", token.ClientId());
        CheckStringField("OAuth token", ctx, input, args, "token_id", token.TokenId());
        CheckStringField("OAuth token", ctx, input, args, "token_id_time", token.TokenIdTime());
        CheckStringField("OAuth token", ctx, input, args, "device_id", token.DeviceId());
        CheckStringField("OAuth token", ctx, input, args, "xtoken_id", token.XtokenId());
        CheckStringField("OAuth token", ctx, input, args, "xtoken_shard", IntToString<10>(token.XtokenShard()));
        CheckStringField("OAuth token", ctx, input, args, "meta", token.Meta());
        CheckStringField("OAuth token", ctx, input, args, "login_id", token.LoginId());
        CheckStringField("OAuth token", ctx, input, args, "payment_auth_context_id", token.PaymentAuthContextId());
        CheckStringField("OAuth token", ctx, input, args, "payment_auth_scope_addendum", token.PaymentAuthScopeAddendum());

        it = args.find("scopes");
        if (it != args.end()) {
            NAuth::TOAuthToken::TScopeIds result_scopes;
            for (const auto& s : NUtils::ToVector(it->second.Value(), ',')) {
                unsigned scope_id = NUtils::ToUInt(s, "oauth_scope");
                result_scopes.insert(scope_id);
            }
            const NAuth::TOAuthToken::TScopeIds& token_scopes = token.Scopes();
            if (token_scopes != result_scopes) {
                TString str_result_scopes;
                for (const auto& s : result_scopes) {
                    str_result_scopes.append(IntToString<10>(s)).push_back(',');
                }
                if (!str_result_scopes.empty()) {
                    str_result_scopes.pop_back();
                }

                TString str_token_scopes;
                for (const auto& s : token_scopes) {
                    str_token_scopes.append(IntToString<10>(s)).push_back(',');
                }
                if (!str_token_scopes.empty()) {
                    str_token_scopes.pop_back();
                }

                Cout << "Token sc =" << str_token_scopes << "; res_sc=" << str_result_scopes << Endl;

                CheckField("OAuth token", input, "scopes", str_result_scopes, str_token_scopes);
            }
        }

        return true;
    }
}
