#include "check_sessionid.h"

#include "common.h"

#include <passport/infra/tools/ylast/src/func.h>
#include <passport/infra/tools/ylast/src/context/sess_context.h>
#include <passport/infra/tools/ylast/src/utils/session_utils.h>

#include <unordered_map>

namespace NPassport::NLast::NCheck {
    static const std::unordered_map<TString, NAuth::TSession::EStatus> VALID_STATUSES = {
        {"valid", NAuth::TSession::VALID},
        {"need_reset", NAuth::TSession::NEED_RESET},
    };

    bool CheckSessionId(const TTestContext& ctx, const TString& input, const NPassport::NLast::TFunctionArgs& args) {
        TSessContext::GetInstance();

        TSessionContext sessCtx(args);
        NPassport::NLast::TFunctionArgs::const_iterator it;

        // Locate 'type' argument
        it = args.find("type");
        if (it == args.end()) {
            throw TLastError() << "CheckSessionId(): no 'type' argument";
        }

        // Try parse cookie
        NPassport::NAuth::TSession s(TSessContext::GetInstance().Parser().ParseCookie(input, sessCtx.Host));

        auto checkValidCookie = [&](const TString& expectedStatus) {
            auto itStatus = VALID_STATUSES.find(expectedStatus);
            Y_ENSURE(itStatus != VALID_STATUSES.end());

            CheckField("Session cookie", input, "status", itStatus->second, s.IsValid());
            CheckField("Session cookie", input, "version", sessCtx.Version, s.Version());
            CheckField("Session cookie", input, "uid", sessCtx.Uid, s.Uid());

            it = args.find("usercount");
            if (it != args.end()) {
                CheckField("Session cookie", input, "usercount", it->second.Value(), IntToString<10>(s.UserCount()));
            }

            // Check flag
            if ((s.IsPermanent() && sessCtx.Ttl != "1" && sessCtx.Ttl != "5") ||
                (s.Is2Weeks() && sessCtx.Ttl != "3") ||
                (s.IsRestricted() && sessCtx.Ttl != "4") ||
                (s.IsSession() && sessCtx.Ttl != s.Ttl())) {
                CheckField("Session cookie", input, "ttl", sessCtx.Ttl, s.Ttl());
            }

            it = args.find("safe");
            if (it != args.end()) {
                CheckField("Session cookie", input, "safe flag", it->second.Value(), TString(s.Safe() ? "1" : "0"));
            }

            CheckStringField("Session cookie", ctx, input, args, "authid", s.AuthId());
            CheckStringField("Session cookie", ctx, input, args, "keyspace", s.Domsuff());
            CheckStringField("Session cookie", ctx, input, args, "login_id", s.LoginId());

            // allow max 5 users in test cookie
            TString suffixes[] = {"", "_1", "_2", "_3", "_4"};

            for (unsigned idx = 0; idx < sizeof(suffixes) / sizeof(TString) && idx < s.UserCount(); ++idx) {
                TString str_idx("[");
                str_idx.append(IntToString<10>(idx)).push_back(']');

                it = args.find("uid" + suffixes[idx]);
                if (it != args.end()) {
                    CheckField("Session cookie", input, "uid" + str_idx, it->second.Value(), s.Uid(idx));
                }

                it = args.find("glogouted" + suffixes[idx]);
                if (it != args.end()) {
                    CheckField("Session cookie", input, "glogouted" + str_idx, it->second.Value(), TString(s.IsGlogouted(idx) ? "1" : "0"));
                }

                it = args.find("have_pwd" + suffixes[idx]);
                if (it != args.end()) {
                    CheckField("Session cookie", input, "have_password" + str_idx, it->second.Value(), TString(s.HavePassword(idx) ? "1" : "0"));
                }
            }
        };

        // Perform various checks based on the cookie type
        if (it->second.Value() == "noauth") {
            CheckField("Session cookie", input, "status", NPassport::NAuth::TSession::NOAUTH, s.IsValid());
        } else if (VALID_STATUSES.contains(it->second.Value())) {
            checkValidCookie(it->second);
        } else {
            throw TLastError() << "CheckSessionId(): unknown argument value 'type=" << it->second.Value() << "'";
        }

        // Now check cookie time
        if (!TimesClose(s.Age(), sessCtx.Age())) {
            // use "checkField" to fail the check and report error
            CheckField("Session cookie", input, "age", s.Age(), sessCtx.Age());
        }

        return true;
    }
}
