#include "config.h"

#include "engine.h"
#include "func.h"

#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/system/env.h>

#include <fstream>
#include <memory>
#include <sstream>
#include <unordered_map>

namespace NPassport::NLast {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    static TConfig config;

    const TConfig& TConfig::Get() {
        return config;
    }

    TConfig& TConfig::GetMutable() {
        return config;
    }

    void TConfig::Init(const TString& file) {
        config.Parse(file);
    }

    TConfig::TConfig() = default;

    TConfig::~TConfig() = default;

    static const std::unordered_map<TString, NAuth::EEnitityType> TYPE_MAPPING = {
        {"oauth", NAuth::EEnitityType::OAuth},
        {"session", NAuth::EEnitityType::Session},
        {"smth", NAuth::EEnitityType::Smth},
    };

    static NAuth::EEnitityType CastEntityType(const TString& type) {
        auto it = TYPE_MAPPING.find(type);
        Y_ENSURE(it != TYPE_MAPPING.end(),
                 "unknown type: '" << type << "'");
        return it->second;
    }

    static NAuth::TGammaKeeperSettings ParseGammaSettings(const NXml::TConfig& config) {
        NAuth::TGammaKeeperSettings res;

        for (const TString& path : config.SubKeys("/config/gamma_keeper/serve_kspaces/entry")) {
            NAuth::EEnitityType type = CastEntityType(config.AsString(path + "/@type"));

            res.SignedTypes.insert(type);
            res.KeyspaceToType.emplace(
                NAuth::TSessionUtils::Dots2under(config.AsString(path + "/@id")),
                type);
        }

        for (const TString& path : config.SubKeys("/config/gamma_keeper/gamma")) {
            const NAuth::TGammaId id = config.AsNum<NAuth::TGammaId>(path + "/@id");

            NAuth::TGamma gamma{
                .Id = id,
                .Body = std::make_shared<NSecretString::TSecretString>(
                    NUtils::Base64ToBin(config.AsString(path))),
            };
            for (const auto& [keyspace, type] : res.KeyspaceToType) {
                gamma.TypesToCheck.insert(type);
            }

            res.Gammas.emplace(id, std::move(gamma));
        }

        for (const auto& [id, gamma] : res.Gammas) {
            res.SigningGamma = std::max(res.SigningGamma, id);
        }

        return res;
    }

    static TString GetEnvString(const NXml::TConfig& config, const TString& path) {
        const TString val = config.AsString(path, "");
        return val.StartsWith("$") ? GetEnv(TString(val.substr(1))) : val;
    }

    void TConfig::Parse(const TString& file) {
        const NXml::TConfig config = NXml::TConfig::ReadFromFile(file);

        // Passport database access variables
        ParseDb(config, "/config/passportdb", PasspDb);

        // Tvm data
        ParseDb(config, "/config/tvmdb", TvmDb);

        UrlOauth = config.AsString("/config/oauth/url", "");

        TvmPublicKeysUrl = config.AsString("/config/tvm_public_keys", "");

        XServiceTicket.TvmHost = config.AsString("/config/x_service_ticket/tvm_host", "");
        XServiceTicket.TvmPort = config.AsInt("/config/x_service_ticket/tvm_port", 0);
        XServiceTicket.Dst = config.AsNum<ui32>("/config/x_service_ticket/dst", 0);

        for (const TString& path : config.SubKeys("/config/x_service_ticket/client")) {
            const ui32 src = config.AsNum<ui32>(path + "/@src");
            const TString secret = config.AsString(path);
            XServiceTicket.Clients.emplace(src, secret);
        }

        if (config.Contains("/config/x_user_ticket")) {
            XUserTicket = TXUserTicket{
                .TvmSecret = config.AsString("/config/x_user_ticket/tvm_secret"),
                .TvmId = config.AsNum<ui32>("/config/x_user_ticket/tvm_id"),
            };
        }

        for (const TString& path : config.SubKeys("/config/serve_guard_spaces/guard_space")) {
            GuardSpaces.push_back(
                TGuardSpace{
                    .Id = config.AsString(path + "/@id"),
                    .Name = config.AsString(path + "/@name"),
                    .Hostnames = NUtils::ToVector(config.AsString(path + "/@hostnames"), ';'),
                });
        }

        if (config.Contains("/config/gamma_keeper")) {
            GammaKeeperSettings = ParseGammaSettings(config);
        }

        BlackboxHost = GetEnvString(config, "/config/blackbox_host");
        if (BlackboxHost.empty()) {
            BlackboxHost = "localhost";
        }
    }

    void TConfig::ParseDb(const NXml::TConfig& config, const TString& path, TDbConf& db) {
        db.DbDriver = config.AsString(path + "/driver", "");
        db.DbHost = config.AsString(path + "/host", "");
        db.DbPort = config.AsNum<ui16>(path + "/port", 0);
        db.DbUser = GetEnvString(config, path + "/user");
        db.DbPassword = GetEnvString(config, path + "/password");
        db.DbName = config.AsString(path + "/dbname", "");
        db.KeyPath = config.AsString(path + "/key", "");
    }
}
