#include "connection.h"

#include "func.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>

namespace NPassport::NLast {
    void TPerformResult::Clear() {
        Output.clear();
        InHdr.clear();
        Page.clear();
    }

    TConnection::TConnection() {
        // Keep trace of what is going on in our own callback.
        // Suppress normal output -- we don't need it.
        // Individual test cases will be able to do extra checks
        // through the hook mechanism.
        Handle_.Set(CURLOPT_VERBOSE, true);
        Handle_.Set(DebugCallback);
        Handle_.Set(CURLOPT_DEBUGDATA, &PerfRes_);
        Handle_.Set(WriteCallback);
        Handle_.Set(CURLOPT_WRITEDATA, &PerfRes_);
        Handle_.Set(HeadersCallback);
        Handle_.Set(CURLOPT_WRITEHEADER, &PerfRes_);

        Handle_.Set(CURLOPT_NOPROGRESS, true);
        Handle_.Set(CURLOPT_FORBID_REUSE, false);
        Handle_.Set(CURLOPT_FRESH_CONNECT, false);

        // Turn on curl's cookie engine
        Handle_.Set(CURLOPT_COOKIEFILE, "");

        // Turn off SSL certificate verification
        Handle_.Set(CURLOPT_SSL_VERIFYPEER, false);
        Handle_.Set(CURLOPT_SSL_VERIFYHOST, false);
    }

    const TPerformResult& TConnection::Perform(const TTestContext& ctx) {
        PerfRes_.Clear();

        // Path
        TString url = MakeUrl(ctx);
        TString request = MakeRequest(ctx);

        if (ctx.IsPost) {
            Handle_.Set(CURLOPT_CUSTOMREQUEST, nullptr);
            Handle_.Set(CURLOPT_POST, true);
            Handle_.Set(CURLOPT_POSTFIELDS, request);
            Handle_.Set(CURLOPT_POSTFIELDSIZE, request.size());
        } else if (!ctx.CustomMethod.empty()) {
            Handle_.Set(CURLOPT_CUSTOMREQUEST, ctx.CustomMethod.c_str());
            Handle_.Set(CURLOPT_POST, false);
        } else {
            Handle_.Set(CURLOPT_CUSTOMREQUEST, nullptr);
            Handle_.Set(CURLOPT_POST, false);
            if (!request.empty()) {
                url.push_back('?');
                url.append(request);
            }
        }
        Handle_.Set(CURLOPT_URL, url);

        // Headers
        NCurlwrap::TSlist headers;
        MakeHeaders(ctx, headers);
        Handle_.Set(CURLOPT_HTTPHEADER, headers.Ptr());

        // Cookies
        // First, use (documented) black magic to delete all currently known cookies
        Handle_.Set(CURLOPT_COOKIELIST, "ALL");
        TString cookies = MakeCookies(ctx);
        Handle_.Set(CURLOPT_COOKIE, cookies.c_str());

        // Set up redirection
        if (ctx.RedirectCount > 0) {
            Handle_.Set(CURLOPT_MAXREDIRS, ctx.RedirectCount);
        }

        // Go
        Handle_.Perform();
        Handle_.GetInfo(CURLINFO_RESPONSE_CODE, &PerfRes_.HttpCode);
        PerfRes_.Request = std::move(request);

        return PerfRes_;
    }

    TString TConnection::MakeUrl(const TTestContext& ctx) {
        TString url;
        url.reserve(ctx.UrlBase.size() + ctx.Path.size());
        url.append(ctx.UrlBase);

        if (url.back() != '/' && (ctx.Path.empty() || ctx.Path.front() != '/')) {
            url.push_back('/');
        }
        // Don't URL-encode the path because libcurl encodes '/' as well;
        // on the other hand we don't use paths that require encoding yet.
        //        url.append (handle_.UrlEncode (path));
        url.append(ctx.Path);
        if (url.back() != '/') {
            url.push_back('/');
        }

        return url;
    }

    TString TConnection::MakeRequest(const TTestContext& ctx) {
        TString request;
        request.reserve(512);

        for (const auto& pair : ctx.Cgis) {
            if (!request.empty()) {
                request.push_back('&');
            }
            request.append(pair.first);
            request.push_back('=');
            request.append(NUtils::Urlencode(pair.second));
        }

        return request;
    }

    void TConnection::MakeHeaders(const TTestContext& ctx, NCurlwrap::TSlist& headers) {
        for (const auto& pair : ctx.Headers) {
            TString hdr;
            hdr.reserve(pair.second.size() + 2 + pair.first.size());
            hdr.append(pair.first).append(": ");

            if (!pair.second.empty()) {
                hdr.append(pair.second);
            } else {
                hdr.append("\"\"");
            }

            headers.Append(hdr);
        }
    }

    TString TConnection::MakeCookies(const TTestContext& ctx) {
        TString cookies;
        cookies.reserve(64 * ctx.Cookies.size());
        bool first = true;

        for (const auto& pair : ctx.Cookies) {
            if (!first) {
                cookies.append("; ");
            } else {
                first = false;
            }
            cookies.append(pair.first);
            cookies.push_back('=');
            cookies.append(pair.second);
        }

        return cookies;
    }

    int TConnection::DebugCallback(
        CURL*,              /* the handle/transfer this concerns */
        curl_infotype type, /* what kind of data */
        char* data,         /* points to the data (not-NULL terminated) */
        size_t size,        /* size of the data pointed to */
        void* userptr) {
        TPerformResult* ins = (TPerformResult*)userptr;

        switch (type) {
            case CURLINFO_HEADER_OUT:
                ins->Output.append(data, size);
                break;
            default:
                break;
        }

        return 0;
    }

    size_t TConnection::WriteCallback(
        char* buffer,
        size_t size,
        size_t nitems,
        void* outstream) {
        TPerformResult* ins = (TPerformResult*)outstream;
        ins->Page.append(buffer, size * nitems);
        return size * nitems;
    }

    size_t TConnection::HeadersCallback(
        void* header,
        size_t size,
        size_t nitems,
        void* arg) {
        TPerformResult* ins = (TPerformResult*)arg;
        ins->InHdr.append((const char*)header, size * nitems);
        return size * nitems;
    }

}
