#include "ticket_context.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/tvm/common/decryptor.h>
#include <passport/infra/libs/cpp/tvm/common/private_key.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

#include <contrib/libs/openssl/include/openssl/evp.h>
#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/deprecated/service_context.h>
#include <library/cpp/tvmauth/src/utils.h>

#include <ctime>

namespace NPassport::NLast {
    TTicketContext& TTicketContext::GetInstance() {
        static TTicketContext ins(TConfig::Get());
        return ins;
    }

    const TString& TTicketContext::TvmPublicKeys() const {
        return TvmPublicKeys_;
    }

    const TString& TTicketContext::ServiceTicket(NTvmAuth::TTvmId clientId) const {
        static const TString EMPTY_;
        auto it = ServiceTickets_.find(clientId);
        if (it == ServiceTickets_.end()) {
            return EMPTY_;
        }
        return it->second;
    }

    const NTvmCommon::TPrivateKey& TTicketContext::GetPrivateKeys() const {
        if (!PrivKeys_) {
            throw yexception() << "tvm_common::PrivateKey is not initialized";
        }
        return *PrivKeys_;
    }

    TTicketContext::TTicketContext(const TConfig& config) {
        if (!config.TvmDb.DbHost.empty()) {
            InitViaDb(config.TvmDb, Secrets_, OldSecrets_);
        }

        if (!config.XServiceTicket.TvmHost.empty()) {
            TSimpleHttpClient cl(config.XServiceTicket.TvmHost, config.XServiceTicket.TvmPort);
            TStringStream out;
            cl.DoGet(TString("/2/keys?lib_version=") + NTvmAuth::LibVersion(), &out);
            TvmPublicKeys_ = out.Str();
        }

        if (!config.XServiceTicket.Clients.empty()) {
            GetServiceTickets(config);
        }

        if (config.XUserTicket) {
            NTvmAuth::NTvmApi::TClientSettings settings;
            settings.SetTvmHostPort(config.XServiceTicket.TvmHost, config.XServiceTicket.TvmPort);
            settings.SetSelfTvmId(config.XServiceTicket.Dst);

            NTvmAuth::NTvmApi::TClientSettings::TDstMap dsts{
                {NTvmCommon::TServiceTickets::TVMAPI_, config.XUserTicket->TvmId},
            };
            settings.EnableServiceTicketsFetchOptions(NUtils::ReadFile(config.XUserTicket->TvmSecret), std::move(dsts));

            std::shared_ptr<NTvmAuth::TTvmClient> tvmClient = std::make_shared<NTvmAuth::TTvmClient>(settings, NTvmLogger::TLogger::Create());

            PrivKeys_ = std::make_unique<NTvmCommon::TPrivateKey>(
                NTvmCommon::TPrivateKeySettings{
                    .TvmHost = settings.TvmHost,
                    .TvmPort = settings.TvmPort,
                    .EnableDiskCache = false,
                    .PreferredKeyIdx = 5,
                    .Env = NTvmAuth::EBlackboxEnv::Test,
                },
                tvmClient);
        }
    }

    void TTicketContext::InitViaDb(const TConfig::TDbConf& conf,
                                   TSecrets& secrets,
                                   TSecrets& oldSecrets) {
        NTvmCommon::TDecryptor decr(NUtils::ReadFile(conf.KeyPath));

        NDbPool::TDestinationPtr dsn = NDbPool::TDestination::Create(
            conf.DbDriver,
            conf.DbUser,
            conf.DbPassword,
            conf.DbName,
            "",
            "",
            {{"ssl_mode", "SSL_MODE_DISABLED"}});

        NDbPool::TDbPool pool(NDbPool::TDbPoolSettings{
            .Dsn = dsn,
            .Hosts = {{
                conf.DbHost,
                conf.DbPort,
            }},
            .Size = 5,
        });
        std::unique_ptr<NDbPool::TResult> res;

        {
            NDbPool::TBlockingHandle sqlh(pool);
            res = sqlh.Query("SELECT id, value FROM tvm_client_attributes WHERE type=5 ORDER BY id");
        }
        ParseSecrets(*res, decr, secrets);

        {
            NDbPool::TBlockingHandle sqlh(pool);
            res = sqlh.Query("SELECT id, value FROM tvm_client_attributes WHERE type=19 ORDER BY id");
        }
        ParseSecrets(*res, decr, oldSecrets);
    }

    void TTicketContext::GetServiceTickets(const TConfig& config) {
        const TString ts = IntToString<10>(time(nullptr));

        for (const auto& p : config.XServiceTicket.Clients) {
            const TString dst = IntToString<10>(config.XServiceTicket.Dst);
            TString post = "grant_type=client_credentials";
            post.append("&ts=").append(ts);
            post.append("&dst=").append(dst);
            post.append("&src=").append(IntToString<10>(p.first));

            NTvmAuth::TServiceContext ctx(NUtils::ReadFile(p.second), p.first, TvmPublicKeys_);
            post.append("&sign=").append(ctx.SignCgiParamsForTvm(
                ts,
                dst,
                ""));

            TSimpleHttpClient cl(config.XServiceTicket.TvmHost, config.XServiceTicket.TvmPort);
            TStringStream out;
            cl.DoPost("/2/ticket", post, &out);
            TString json = out.Str();

            rapidjson::Document doc;
            if (!NJson::TReader::DocumentAsObject(json, doc)) {
                throw yexception() << "Failed to parse TVM response: " << json;
            }
            const TString point = "/" + dst + "/ticket";
            rapidjson::Value* jsonValue = rapidjson::Pointer(point.c_str()).Get(doc);
            if (!jsonValue) {
                throw yexception() << "Failed to get ticket: " << json;
            }
            ServiceTickets_.emplace(p.first, jsonValue->GetString());
        }
    }

    void TTicketContext::ParseSecrets(NDbPool::TResult& res,
                                      const NTvmCommon::TDecryptor& decr,
                                      TSecrets& secrets) {
        for (const NDbPool::TRow& row : res.Table()) {
            TString encrVal = decr.DecryptAes(row[1].AsString(), "secret", row[0].AsInt());
            secrets.insert(std::make_pair(
                row[0].AsString(),
                NUtils::Base64url2bin(encrVal)));
        }
    }

}
