#include "engine.h"

#include "check_processor.h"
#include "config.h"
#include "error.h"
#include "utils/sqspace.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/rapidjson/include/rapidjson/document.h>
#include <contrib/libs/rapidjson/include/rapidjson/prettywriter.h>

#include <util/generic/string.h>
#include <util/stream/format.h>
#include <util/string/split.h>

#include <algorithm>
#include <functional>
#include <regex>
#include <unordered_set>

namespace NPassport::NLast {
    //
    // Engine
    //
    TEngine::TEngine()
        : Workers_(TConfig::Get().Threads)
    {
    }

    TEngine::~TEngine() = default;

    bool TEngine::Execute(const TScenario& scen) {
        TTimeStatPrinter time_printer("Test");

        // Iterate over nodes defined in the scenario
        for (const std::unique_ptr<TNode>& node : scen.Nodes()) {
            if (!TestNode(*node, scen)) {
                Cout << Endl
                     << "Bailing out." << Endl;
                return false;
            }
        }

        // If we've run through completion, output final status
        Cout << "Test complete";
        if (Failures_ > 0) {
            Cout << " (" << Failures_ << " failures)";
        }
        Cout << "." << Endl;
        return true;
    }

    bool TEngine::TestNode(const TNode& node, const TScenario& scen) {
        if (node.AllowSingleThreadOnly()) {
            Workers_.Resize(1);
        } else {
            Workers_.Resize(TConfig::Get().Threads);
        }

        // Show progress
        Cout << Endl
             << "*** Testing node: ";
        if (!node.Description().empty()) {
            Cout << node.Description();
        } else {
            Cout << "<undefined>";
        }
        Cout << Endl;

        // if pre-test script is specified, run it
        if (!node.PreTest().empty()) {
            TLog::Debug() << "Engine: pre_test script: started";
            if (system(node.PreTest().c_str())) {
                Cout << "Failed to run pre-test script: '" << node.PreTest() << "'" << Endl;
                return false;
            }
            TLog::Debug() << "Engine: pre_test script: finished";
        }

        // Iterate over node's URLs and test cases. Pass curl handle having all common
        // options set -- it will copied for every case so that individual options
        // can be set on a per-case basis.
        for (const TString& u : node.Urls()) {
            Cout << "*** " << u << Endl;
            for (const auto& c : node.Cases()) {
                if (!TestCase(u, *c)) {
                    // if post-test script is specified, run it
                    if (!node.PostTest().empty()) {
                        TLog::Debug() << "Engine: post_test script: started";
                        if (system(node.PostTest().c_str())) {
                            Cout << "Failed to run post-test script: '" << node.PostTest() << "'" << Endl;
                        }
                        TLog::Debug() << "Engine: post_test script: finished";
                    }
                    return false;
                }
            }
        }

        if (TConfig::Get().PrintUnused) {
            Cout << Endl;
            std::unordered_set<TString> mayBeUsed;
            for (const auto& c : node.Cases()) {
                for (size_t idx = 0; idx < c->Checks().size(); ++idx) {
                    const TResult* res = c->Checks().at(idx)->GetResult();
                    mayBeUsed.insert(res->GetName());
                    if (res && !res->WasUsed()) {
                        Cout << "Unused result: " << c->Description() << " : " << idx << " : " << res->GetName() << Endl;
                    }
                }
            }
            for (const TString& resName : scen.GetAllResults()) {
                if (mayBeUsed.find(resName) == mayBeUsed.end()) {
                    Cout << "Unused result: " << resName << Endl;
                }
            }
        }

        // if post-test script is specified, run it
        if (!node.PostTest().empty()) {
            if (system(node.PostTest().c_str())) {
                Cout << "Failed to run post-test script: '" << node.PostTest() << "'" << Endl;
                return false;
            }
        }

        Cout << Endl;

        return true;
    }

    bool TEngine::TestCase(const TString& urlBase, const TCase& testCase) {
        Cout << "\nExecuting test case: '";
        SquezeSpace(testCase.Description(), Cout) << "'\n";

        // Iterate over case's variable combinations. For every combination
        // defined in the scenario do the following:
        //  - set URL as a concatenation of the base URL, path and all of
        //    CGI parameters
        //  - set HTTP headers
        //  - set cookies
        //  - set misc. curl option
        //  - set test hooks
        //  - invoke HTTP request
        size_t recievedCount = 0;
        TVarSet set(urlBase, testCase);
        const size_t totalCount = set.Count();
        TTestContext ctx;
        for (int setCount = 0; set.Next(ctx); ++setCount) {
            ctx.IsPost = testCase.IsPost();
            ctx.CustomMethod = testCase.CustomMethod();
            ctx.RedirectCount = testCase.Redirect();
            ctx.UrlBase = urlBase;
            ctx.Idx = setCount;
            ctx.Check->GetResult()->SetUsed();

            Workers_.AddTask(std::move(ctx));

            TWorkerResult res;
            if (!Workers_.GetNext(res)) {
                continue;
            }

            if (!ProcResult(res, ++recievedCount, totalCount)) {
                return false;
            }
        }

        for (; recievedCount < totalCount; ++recievedCount) {
            TWorkerResult res;
            while (!Workers_.GetNext(res)) {
                std::this_thread::sleep_for(std::chrono::milliseconds(10));
            }

            if (!ProcResult(res, recievedCount, totalCount)) {
                return false;
            }
        }

        // Print out total number of combinations
        Cout << "Total combinations in this test case: " << totalCount << (TConfig::Get().ShowHttpTrace ? "" : "\r") << Endl;

        return true;
    }

    bool TEngine::ProcResult(TWorkerResult& res, size_t recievedCount, size_t totalCount) {
        if (!res.Res) {
            Cout << res.Output << Endl;
            if (!TConfig::Get().IgnoreErrors) {
                return false;
            }

            ++Failures_;
        } else if (TConfig::Get().ShowHttpTrace) {
            Cout << res.Output;
        }

        if ((!TConfig::Get().ShowHttpTrace || !res.Res) && !TConfig::Get().Quiet) {
            Cout << recievedCount << "/" << totalCount
                 << " (" << Prec(100. * recievedCount / totalCount, PREC_POINT_DIGITS, 1) << "%)\r" << Flush;
        }

        return true;
    }

}
