#pragma once

#include <util/generic/string.h>
#include <util/stream/output.h>
#include <util/system/types.h>

#include <map>
#include <memory>
#include <vector>

namespace NPassport::NLast::NLibretto {
    struct TArg;
    struct TResultItem;
}

namespace NPassport::NLast {
    class TCheck;
    class TVariator;
    //
    // Context used to invoke generate and check functions
    //
    class TTestContext {
    public:
        void Clear();

        using TValueSet = std::multimap<TString, TString>;
        TValueSet Headers;
        TValueSet Cookies;
        TValueSet Cgis;
        TString Path;
        TString Schema;

        TString ErrMsg;

        // For Worker
        const TCheck* Check = nullptr;
        bool IsPost = false;
        TString CustomMethod;
        int RedirectCount = 0;
        TString UrlBase;
        ui64 Idx = 0;
        bool IsNeedSignTs = false;
        bool IsNeedTvm2Sign = false;
        bool IsNeedTvm2SignWithOldSecret = false;
        std::vector<std::shared_ptr<TVariator>> Vars; // TODO std::unique_ptr
    };

    class TDynamicResultItem;

    //
    // Arguments container used by Generate and Check functions
    //
    class TArg {
    public:
        enum EResultModifier {
            None,
            KeyCount,
            ArraySize
        };

        // comp
        static const TString SUB_FUNC;
        static const TString PLAIN_TEXT;
        static const TString REGEX;
        static const TString RELATION;
        static const TString STARTS_WITH;
        static const TString CONTAINS;

        // res_mod
        static const TString NONE;
        static const TString KEY_COUNT;
        static const TString ARRAY_SIZE;

        TArg(const TString& value);

        TArg(const NLibretto::TArg& xarg);

        void SetValue(const TString& string) {
            ExpectedValue_ = string;
        }

        void SetValue(long value) {
            ExpectedValueInt_.first = true;
            ExpectedValueInt_.second = value;
        }

        const TString& Value() const {
            return ExpectedValue_;
        }

        operator const TString&() const {
            return ExpectedValue_;
        }

        bool Match(const TString& value, const TTestContext& ctx) const;

        EResultModifier ResMod() const {
            return ResMod_;
        }

        TString CheckInfo() const;

        void SetCompRegex();

    private:
        using TCheckFunc = bool (TArg::*)(const TString& value, const TString& expected) const;
        using TPreCompFunc = TString (*)(const TString& value);

        void SetComp(const TString& comp, const std::optional<ui64> min, const std::optional<ui64> max);
        void SetResMod(const TString& resMod);

        // check - for 'comp'
        bool PlainCheck(const TString& value, const TString& expected) const;
        bool RegexCheck(const TString& value, const TString& expected) const;
        bool RelationCheck(const TString& value, const TString& expected) const;
        bool StartsWithCheck(const TString& value, const TString& expected) const;
        bool ContainsCheck(const TString& value, const TString& expected) const;

        void InitNaturalArg(const NLibretto::TArg& xarg);
        void InitSubFunc(const NLibretto::TResultItem& xarg);

        TCheckFunc CheckFunc_;
        EResultModifier ResMod_;
        bool NeedSubstitute_;
        mutable TString LastFailed_;

        TString ExpectedValue_;
        std::pair<bool, long> ExpectedValueInt_;
        std::pair<bool, long> Min_;
        std::pair<bool, long> Max_;

        std::shared_ptr<TDynamicResultItem> SubFunc_;
    };

    using TFunctionArgs = std::multimap<TString, TArg>;

    //
    // Substitute embedded references to test case variables in the regex with current values.
    //
    // References have the following syntax:
    //
    //  \~`path`
    //  \~`cgi:CGINAME`
    //  \~`cookie:COOKIENAME`
    //
    //  Also, %% may be used to indicate that value must be URL-encoded, such as:
    //
    //  \~`%%cgi:CGINAME`
    //
    TString SubstituteVars(const TString& src, const TTestContext& ctx, bool regexSafe = false);

}

template <>
inline void Out<NPassport::NLast::TArg>(IOutputStream& o, TTypeTraits<NPassport::NLast::TArg>::TFuncParam value) {
    o << value.Value();
}
