#include "gen_oauth_token.h"

#include "common.h"

#include <passport/infra/tools/ylast/src/config.h>
#include <passport/infra/tools/ylast/src/context/oauth_context.h>
#include <passport/infra/tools/ylast/src/utils/session_utils.h>

#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/http/simple/http_client.h>

#include <util/stream/str.h>

namespace NPassport::NLast::NGen {
    static TString GetTokenFromOAuth(const NPassport::NLast::TFunctionArgs& args) {
        TString post_fields;
        for (const auto& pair : args) {
            if (!post_fields.empty()) {
                post_fields.push_back('&');
            }
            post_fields.append(pair.first).push_back('=');
            post_fields.append(NUtils::Urlencode(pair.second.Value()));
        }

        TSimpleHttpClientOptions opts(TConfig::Get().UrlOauth);
        TKeepAliveHttpClient cl(opts.Host(), opts.Port());
        cl.DisableVerificationForHttps();
        TStringStream out;
        Y_ENSURE(200 == cl.DoPost("/token", post_fields, &out),
                 "Host: " << TConfig::Get().UrlOauth << Endl
                          << "Request: " << post_fields << Endl
                          << "Response: " << out.Str() << Endl);
        TString data = out.Str();

        rapidjson::Document d;
        Y_ENSURE_EX(NJson::TReader::DocumentAsObject(data, d),
                    TLastError() << "broken response from OAuth: " << data);

        TString token;
        Y_ENSURE_EX(NJson::TReader::MemberAsString(d, "access_token", token),
                    TLastError() << "Can't get key 'access_token' from OAuth response: "
                                 << data << ". Post body was: " << post_fields);

        TLog::Debug() << "Got token from OAuth: " << token << ". Post body was: " << post_fields;
        return token;
    }

    static TString GenerateStatelessToken(const NPassport::NLast::TFunctionArgs& args) {
        TOAuthContext::GetInstance();

        NPassport::NLast::TFunctionArgs::const_iterator it;

        // Locate uid, time and expire time
        TString uid;

        it = args.find("uid");
        if (it != args.end()) {
            uid = it->second;
        }

        // by default: token created now, expires in one hour
        TString time_delta("0");
        TString expire_delta("3600");

        it = args.find("time");
        if (it != args.end()) {
            time_delta = it->second;
        }

        it = args.find("expires");
        if (it != args.end()) {
            expire_delta = it->second;
        }

        time_t create_time = CalcTimeFromDelta(time_delta);
        NAuth::TOAuthToken token = TOAuthContext::GetInstance().OauthParser->CreateToken(
            uid,
            create_time,
            CalcTimeFromDelta(expire_delta),
            "",
            IntToString<10>(create_time) + "123"); // time in ms, not customized yet

        it = args.find("version");
        if (it != args.end()) {
            token.SetVersion(NUtils::ToUInt(it->second.Value(), "version"));
        }

        it = args.find("client_id");
        if (it != args.end()) {
            token.SetClientId(it->second);
        }

        it = args.find("device_id");
        if (it != args.end()) {
            token.SetDeviceId(it->second);
        }

        it = args.find("xtoken_id");
        if (it != args.end()) {
            token.SetXtokenId(it->second);
        }

        it = args.find("xtoken_shard");
        if (it != args.end()) {
            token.SetXtokenShard(NUtils::ToUInt(it->second.Value(), "xtoken_shard"));
        }

        it = args.find("x_meta");
        if (it != args.end()) {
            token.SetMeta(it->second);
        }

        it = args.find("login_id");
        if (it != args.end()) {
            token.SetLoginId(it->second);
        }

        it = args.find("payment_auth_context_id");
        if (it != args.end()) {
            token.SetPaymentAuthContextId(it->second);
        }

        it = args.find("payment_auth_scope_addendum");
        if (it != args.end()) {
            token.SetPaymentAuthScopeAddendum(it->second);
        }

        it = args.find("scopes");
        if (it != args.end()) {
            NAuth::TOAuthToken::TScopeIds scopes;
            for (const auto& s : NUtils::ToVector(it->second.Value(), ',')) {
                unsigned scope_id = NUtils::ToUInt(s, "oauth_scope");
                scopes.insert(scope_id);
            }
            token.SetScopes(scopes);
        }

        TString err_msg;
        TString token_str = TOAuthContext::GetInstance().OauthParser->MakeTokenStr(token, err_msg);

        if (token_str.empty()) {
            throw TLastError() << "GenerateToken(): couldn't generate token: " << err_msg;
        }

        if (args.find("broken_sign") != args.end()) { // generate test token with broken signature
            char c = token_str[token_str.size() - 2]; // change one base64 letter to another base64
            token_str[token_str.size() - 2] = (c != 'A') ? 'A' : 'B';
        }

        return token_str;
    }

    TString GenerateOAuthToken(const NPassport::NLast::TFunctionArgs& args) {
        // For constant tokens - just return value from scenario
        NPassport::NLast::TFunctionArgs::const_iterator valueIt = args.find("const_value");
        if (valueIt != args.end()) {
            return valueIt->second.Value();
        }

        // For stateless tokens - generate locally
        NPassport::NLast::TFunctionArgs::const_iterator it = args.find("type");
        if (it != args.end() && it->second.Value() == "stateless") {
            return GenerateStatelessToken(args);
        }

        // else it is plain old OAuth token, ask OAuth for token
        return GetTokenFromOAuth(args);
    }
}
