#include "libretto.h"

#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NLast::NLibretto {
    class TSingleElementError: public yexception {
    public:
        TSingleElementError(const TStringBuf xpath) {
            (*this) << "There must be exactly one item in the set '" << xpath << "'";
        }
    };

    class TDuplicationElementError: public yexception {
    public:
        TDuplicationElementError(const TStringBuf xpath) {
            (*this) << "There must be no more than one item in the set '" << xpath << "'";
        }
    };

    static std::vector<TString> Tokenize(const TString& value) {
        return NUtils::ToVector(value, " \t\r\n;");
    }

    TLibretto::TLibretto(const TString& filename)
        : Xml_(NXml::TConfig::ReadFromFile(filename))
    {
    }

    std::vector<TString> TLibretto::GetResultNames() const {
        std::vector<TString> res = Xml_.SubKeys("/test/variables/result");
        for (TString& path : res) {
            path = Xml_.AsString(path + "/@name");
        }
        return res;
    }

    TResult TLibretto::GetResult(const TString& name) const {
        TString path = "/test/variables/result[@name='" + name + "']"; // TODO
        CheckSingle(path);

        TResult res;
        res.Name = name;
        res.StatusCode = Xml_.AsInt(path + "/status-code", 200);
        for (const TString& p : Xml_.SubKeys(path + "/set-cookie")) {
            res.SetCookie.push_back(ParseSetCookie(p));
        }
        for (const TString& p : Xml_.SubKeys(path + "/header")) {
            res.Header.push_back(ParseResultItem(p, EWithName::True));
        }

        if (Xml_.Contains(path + "/body")) {
            CheckSingle(path + "/body");
            res.XmlBody = ParseResultItem(path + "/body", EWithName::False);
        } else if (Xml_.Contains(path + "/xmlBody")) {
            CheckSingle(path + "/xmlBody");
            res.XmlBody = ParseResultItem(path + "/xmlBody", EWithName::False);
        }

        if (Xml_.Contains(path + "/jsonBody")) {
            CheckSingle(path + "/jsonBody");
            res.JsonBody = ParseResultItem(path + "/jsonBody", EWithName::False);
        }

        return res;
    }

    std::vector<TNode> TLibretto::GetNodes() const {
        std::vector<TNode> res;
        for (const TString& path : Xml_.SubKeys("/test/node")) {
            res.push_back(ParseNode(path));
        }
        return res;
    }

    TNamedVar TLibretto::GetNamedVar(const TString& type, const TString& name, const TString& refList) const {
        TNamedVar res;

        const TString path = NUtils::CreateStr("/test/variables/", type, "[@name='", name, "']");
        CheckSingle(path);

        res.ResultItem = ParseResultItem(path, EWithName::True);
        res.NeedPregeneration = Xml_.Contains(path + "/@pregeneration");

        if (refList == "*") {
            for (const TString& p : Xml_.SubKeys(path + "/instance")) {
                res.Intanses.push_back({Xml_.AsString(p + "/@id"), ParseResultItem(p, EWithName::False)});
            }
        } else {
            for (const TString& id : Tokenize(refList)) {
                res.Intanses.push_back({
                    id,
                    ParseResultItem(NUtils::CreateStr(path, "/instance[@id='", id, "']"), EWithName::False),
                });
            }
        }

        return res;
    }

    TResultItem TLibretto::ParseResultItem(const TString& path, EWithName withName) const {
        TResultItem res;
        CheckSingle(path);

        if (EWithName::True == withName) {
            res.Name = GetSingleString(path + "/@name");
        }

        res.Value = Xml_.AsString(path, "");

        res.Func = Xml_.AsString(path + "/@func", "");
        for (const TString& a : Xml_.SubKeys(path + "/arg")) {
            res.Args.push_back(ParseArg(a));
        }

        return res;
    }

    TSetCookie TLibretto::ParseSetCookie(const TString& path) const {
        TSetCookie res;

        res.Name = GetSingleString(path + "/@name");
        res.Ignore = Xml_.Contains(path + "/@dontcare");

        if (res.Ignore) {
            return res;
        }

        CheckSingle(path + "/value");
        res.Value = ParseResultItem(path + "/value", EWithName::False);

        for (const TString& p : Xml_.SubKeys(path + "/attribute")) {
            res.Attribute.push_back(ParseResultItem(p, EWithName::True));
        }

        return res;
    }

    TArg TLibretto::ParseArg(const TString& path) const {
        TArg res;

        res.Name = GetSingleString(path + "/@name", ECanBeEmpty::True);
        res.ExpectedValue = Xml_.AsString(path, "");

        if (Xml_.Contains(path + "/@func")) {
            res.SubItem = std::make_unique<TResultItem>(ParseResultItem(path, EWithName::False));
            return res;
        }

        if (Xml_.Contains(path + "/@comp")) {
            res.Comparator = GetSingleString(path + "/@comp");
            res.CompMin = GetOptionalInt(path + "/@min");
            res.CompMax = GetOptionalInt(path + "/@max");
        }
        if (Xml_.Contains(path + "/@res_mod")) {
            res.ResultModificator = GetSingleString(path + "/@res_mod");
        }

        res.DoesNeedSubstitute = Xml_.AsBool(path + "/@substitute", false);

        return res;
    }

    TNode TLibretto::ParseNode(const TString& path) const {
        TNode res;

        res.Description = Xml_.AsString(path + "/description", "");
        res.SingleThreadOnly = Xml_.AsBool(path + "/single_thread_only", false);
        res.PreTestScript = Xml_.AsString(path + "/pre-test", "");
        res.PostTestScript = Xml_.AsString(path + "/post-test", "");
        res.Url = Tokenize(Xml_.AsString(path + "/url", ""));

        for (const TString& cpath : Xml_.SubKeys(path + "/case")) {
            res.Cases.push_back(ParseCase(cpath));
        }

        return res;
    }

    std::pair<TString, TCase> TLibretto::ParseCase(const TString& path) const {
        TString lastId = Xml_.AsString(path + "/@lastid", "");

        TCase res;

        res.CheckSet = ParseCheckSet(path);

        res.TvmSignTs = Xml_.Contains(path + "/tvm_sign_ts");
        res.Tvm2SignOld = Xml_.Contains(path + "/tvm2_sign__old");
        res.Tvm2Sign = Xml_.Contains(path + "/tvm2_sign");

        res.IsPost = Xml_.Contains(path + "/post");
        res.CustomMethod = Xml_.AsString(path + "/custom_method", "");
        Y_ENSURE(!res.IsPost || !res.CustomMethod,
                 "'post' and 'custom_method' conflicts with each other: " << path);

        res.Redirects = Xml_.AsInt(path + "/redirect", 0);

        for (const TString& c : Xml_.SubKeys(path + "/check")) {
            res.Checks.push_back(ParseCheck(c));
        }

        return {std::move(lastId), std::move(res)};
    }

    TPath TLibretto::ParsePath(const TString& path) const {
        TPath res;

        res.WithNull = WithNull(path);

        const TString value = Xml_.AsString(path);
        if (value == "*") {
            for (const TString& p : Xml_.SubKeys("/test/variables/path/instance")) {
                res.Values.push_back(Xml_.AsString(p));
            }
        } else {
            for (const TString& id : Tokenize(value)) {
                res.Values.push_back(GetSingleString("/test/variables/path/instance[@id='" + id + "']"));
            }
        }

        return res;
    }

    TCheck TLibretto::ParseCheck(const TString& path) const {
        TCheck res;

        res.CheckSet = ParseCheckSet(path);
        res.Result = GetSingleString(path + "/result");

        return res;
    }

    TCheckSet TLibretto::ParseCheckSet(const TString& path) const {
        TCheckSet res;

        res.Description = Xml_.AsString(path + "/description", "");
        for (const TString& p : Xml_.SubKeys(path + "/path")) {
            res.Path.push_back(ParsePath(p));
        }

        for (const TString& p : Xml_.SubKeys(path + "/cgi")) {
            res.Cgi.push_back(ParseVariable(p));
        }
        for (const TString& p : Xml_.SubKeys(path + "/cookie")) {
            res.Cookie.push_back(ParseVariable(p));
        }
        for (const TString& p : Xml_.SubKeys(path + "/header")) {
            res.Header.push_back(ParseVariable(p));
        }

        return res;
    }

    TVariable TLibretto::ParseVariable(const TString& path) const {
        TVariable res;

        res.Name = Xml_.AsString(path + "/@name");
        res.Value = Xml_.AsString(path, "", NXml::TConfig::EFetchContent::All); // skip comment in xml
        res.WithNull = WithNull(path);

        return res;
    }

    bool NPassport::NLast::NLibretto::TLibretto::WithNull(const TString& path) const {
        return Xml_.AsInt(path + "/@withnull", 0);
    }

    template <class T>
    void TLibretto::CheckSingle(const TString& path) const {
        Y_ENSURE_EX(Xml_.SubKeys(path).size() == 1, T(path));
    }

    TString TLibretto::GetSingleString(const TString& path, ECanBeEmpty canBeEmpty) const {
        CheckSingle(path);
        return canBeEmpty == ECanBeEmpty::True ? Xml_.AsString(path, "") : Xml_.AsString(path);
    }

    std::optional<ui64> TLibretto::GetOptionalInt(const TString& path) const {
        if (!Xml_.Contains(path)) {
            return {};
        }

        CheckSingle(path);
        return Xml_.AsNum<ui64>(path);
    }
}
