#include "config.h"
#include "engine.h"
#include "libretto.h"

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/openssl/init/init.h>

#include <util/generic/string.h>

using TScenarioList = std::vector<TString>;
static void ParseCommandLine(int argc, char** argv, TScenarioList& scens, TString& configFile) {
    NLastGetopt::TOpts opts;
    opts.AddHelpOption();
    opts.SetFreeArgsMin(1);
    opts.SetFreeArgDefaultTitle("test scenario file");
    opts.AddLongOption('c', "config", "test config file").Required();
    opts.AddLongOption('p', "print", "print out the scenario in a human-readable form").NoArgument();
    opts.AddLongOption('i', "ignore", "ignore failed test cases").NoArgument();
    opts.AddLongOption('V', "verbose", "verbose (print out complete HTTP communication trace)").NoArgument();
    opts.AddLongOption('C', "case", "@lastid of case to execute (execute all if none specified)");
    opts.AddLongOption('U', "url", "override test scenario URL");
    opts.AddLongOption('N', "unused", "print unused Results").NoArgument();
    opts.AddCharOption('j', "number of threads");
    opts.AddCharOption('q', "quiet print to stdout").NoArgument();
    NLastGetopt::TOptsParseResult res(&opts, argc, argv);

    NPassport::NLast::TConfig& config = NPassport::NLast::TConfig::GetMutable();

    for (const TString& s : res.GetFreeArgs()) {
        scens.push_back(s);
    }
    configFile = res.Get('c');
    config.ShowParsedScen = res.Has('p');
    config.IgnoreErrors = res.Has('i');
    config.ShowHttpTrace = res.Has('V');
    config.Quiet = res.Has('q');
    if (res.Has('C')) {
        config.Cases.insert(res.Get('C'));
    }
    if (res.Has('U')) {
        config.Url = res.Get('U');
    }
    config.PrintUnused = res.Has('N');
    if (res.Has('j')) {
        config.Threads = atoi(res.Get('j'));
    }
}

int main(int argc, char* argv[]) {
    using namespace NPassport::NLast;
    InitOpenSSL();

    try {
        TTimeStatPrinter time_printer("Runtime");

        TString configFile;
        TScenarioList scenList;

        // Set globals
        ParseCommandLine(argc, argv, scenList, configFile);

        // Parse config file (if specified)
        if (!configFile.empty()) {
            TConfig::Init(configFile);
        }

        // Initialize libcurl
        NCurlwrap::TSession curlSession;

        for (const TString& s : scenList) {
            Cout << "======== " << s << " ========" << Endl;

            NLibretto::TLibretto libretto(s);

            // Parse scenario and print out results in a human-readable form
            TScenario scen(libretto);
            if (TConfig::Get().ShowParsedScen) {
                scen.Print();
            }
            // Go...
            TEngine eng;
            if (!eng.Execute(scen)) {
                return 1;
            }
        }

    } catch (const std::exception& e) {
        Cout << "!!! TEST FAIL !!!" << Endl;
        Cout << "Exception: '" << e.what() << "'" << Endl;
        return 2;
    }
    return 0;
}
