#include "result.h"
#include "scenario.h"
#include "variables.h"

#include <util/generic/string.h>

namespace NPassport::NLast {
    //
    // Misc helpers
    //
    static TString FormatFunctionCall(const TString& fname, const TFunctionArgs& args) {
        TString callstr;
        callstr.reserve(100);
        callstr.append(fname).append("(");
        bool first = true;
        for (const auto& pair : args) {
            if (!first) {
                callstr.append(",");
            } else {
                first = false;
            }
            callstr.append(pair.first).append("=").append(pair.second.Value());
        }
        callstr.append(")");

        return callstr;
    }

    //
    // Variables
    //
    void TNamedVar::Print(IOutputStream& s) const {
        s << Name_ << ": ";
        bool first = true;
        for (const auto& val : Values_) {
            if (!first) {
                s << ", ";
            } else {
                first = false;
            }

            if (val) {
                val->Print(s);
            } else {
                s << "NULL";
            }
        }
        s << Endl;
    }

    void TCookieVar::Print(IOutputStream& s) const {
        s << "Cookie: ";
        TNamedVar::Print(s);
    }

    void TCgiVar::Print(IOutputStream& s) const {
        s << "CGI: ";
        TNamedVar::Print(s);
    }

    void THeaderVar::Print(IOutputStream& s) const {
        s << "Header: ";
        TNamedVar::Print(s);
    }

    void TPathVar::Print(IOutputStream& s) const {
        s << "Path: ";
        bool first = true;
        for (const auto& val : Values_) {
            if (!first) {
                s << ", ";
            } else {
                first = false;
            }
            if (val) {
                s << "'" << *val << "'";
            } else {
                s << "NULL";
            }
        }
        s << Endl;
    }

    //
    // Variators
    //
    void TNamedVariator::Print(IOutputStream& s) const {
        s << "Variator: " << Name_;
        if (*Current_ != nullptr) {
            s << " ( " << (*Current_)->Id() << " ) # ";
            (*Current_)->Print(s);
        } else {
            s << "NULL";
        }
        s << Endl;
    }

    void TCgiVariator::Print(IOutputStream& s) const {
        s << "CGI ";
        TNamedVariator::Print(s);
    }

    void THeaderVariator::Print(IOutputStream& s) const {
        s << "HDR ";
        TNamedVariator::Print(s);
    }

    void TCookieVariator::Print(IOutputStream& s) const {
        s << "COOKIE ";
        TNamedVariator::Print(s);
    }

    void TPathVariator::Print(IOutputStream& s) const {
        s << "PATH Variator: ";
        if (*Current_ != nullptr) {
            s << *(*Current_);
        } else {
            s << "NULL";
        }
        s << Endl;
    }

    void TVarSet::Print(IOutputStream& s) const {
        for (const auto& v : Variators_) {
            if (v) {
                v->Print(s);
            }
        }
    }

    //
    // Scenario
    //
    void TCheck::Print(IOutputStream& s) const {
        s << "    Check: " << Description_ << Endl
          << "    -----" << Endl;

        if (Path_ != nullptr) {
            s << "      ";
            Path_->Print(s);
        }

        for (const auto& c : Cgi_) {
            s << "      ";
            c->Print(s);
        }

        for (const auto& h : Header_) {
            s << "      ";
            h->Print(s);
        }

        for (const auto& cookie : Cookie_) {
            s << "      ";
            cookie->Print(s);
        }

        if (Result_ != nullptr) {
            Result_->Print(s);
        }
    }

    void TCase::Print(IOutputStream& s) const {
        s << "  Case" << Endl
          << "  ----" << Endl;
        s << "    Description: '" << Description_ << "'" << Endl;
        s << "    Redirect: " << Redirect_ << Endl;

        for (const auto& var : Variables_) {
            s << "    ";
            var->Print(s);
        }

        for (const auto& c : Checks_) {
            c->Print(s);
        }
    }

    void TNode::Print(IOutputStream& s) const {
        s << "NODE" << Endl
          << "----" << Endl;
        s << "  URLs: ";
        for (size_t i = 0; i < Urls_.size(); ++i) {
            if (i > 0) {
                s << ", ";
            }
            s << "'" << Urls_[i] << "'";
        }
        s << Endl
          << "  Description: '" << Description_ << "'" << Endl;

        for (const auto& c : Cases_) {
            c->Print(s);
        }
    }

    void TScenario::Print(IOutputStream& s) const {
        for (const auto& node : Nodes_) {
            node->Print(s);
        }
    }

    //
    // Result
    //
    TString TDynamicResultItem::GetValue() const {
        return FormatFunctionCall(Function_, Args_);
    }

    void TStaticNamedInst::Print(IOutputStream& s) const {
        s << "'" << Value_ << "' (id=" << Id_ << ")";
    }

    void TDynamicNamedInst::Print(IOutputStream& s) const {
        s << FormatFunctionCall(Function_, Args_) << " (id=" << Id_ << ")";
    }

    void TResultCookie::Print(IOutputStream& s) const {
        s << "        Cookie: " << Name_ << '=';

        if (Value_ != nullptr) {
            s << '\'' << Value_->GetValue() << '\'';
            for (const auto& pair : Attribs_) {
                s << "; " << pair.first << "='" << pair.second->GetValue() << "'";
            }
        } else {
            s << "DONT CARE";
        }

        s << Endl;
    }

    void TResult::Print(IOutputStream& s) const {
        s << "      Result: " << Name_ << Endl
          << "      ------" << Endl;
        s << "        Status-code: " << StatusCode_ << Endl;

        for (const auto& cookie : Cookies_) {
            cookie->Print(s);
        }

        s << "        Xml body: ";
        if (XmlBody_ != nullptr) {
            s << XmlBody_->GetValue();
        } else {
            s << "none";
        }
        s << Endl;

        s << "        Json body: ";
        if (JsonBody_ != nullptr) {
            s << JsonBody_->GetValue();
        } else {
            s << "none";
        }
        s << Endl;
    }

}
