#pragma once

#include "error.h"
#include "func.h"
#include "checks/facade.h"
#include "generators/facade.h"

#include <util/generic/string.h>

#include <map>
#include <memory>
#include <vector>

namespace NPassport::NLast::NLibretto {
    struct TResultItem;
}

namespace NPassport::NLast {
    //
    // Abstract base for a result item value which may be static
    // (hardcoded in the configuration) or dynamic (verified at
    // run time by calling a check function).
    //
    class TResultItem {
    public:
        TResultItem() = default;
        virtual ~TResultItem() = default;

        virtual TString GetValue() const = 0;
        virtual bool Match(const TTestContext& ctx, const TString&, bool canThrowMatchError = false) const = 0;
    };

    class TStaticResultItem: public TResultItem {
    public:
        TStaticResultItem(const TString& value)
            : Value_(value)
        {
        }
        ~TStaticResultItem() override = default;

        TString GetValue() const override {
            return Value_;
        }
        bool Match(const TTestContext& ctx, const TString& v, bool canThrowMatchError) const final;

    private:
        TString Value_;
    };

    //
    // Dynamic result item (value is verified at run time by calling
    // a check function).
    //
    class TDynamicResultItem: public TResultItem {
    public:
        TDynamicResultItem(const NLibretto::TResultItem& xitem);
        ~TDynamicResultItem() override;

        TString GetValue() const override;

        bool Match(const TTestContext& ctx, const TString& v, bool canThrowMatchError = false) const final {
            // This quick and dirty hack: need to to review result-check
            // mismatch reporting as a whole.
            try {
                return Invoke(ctx, v);
            } catch (const TMatchError& e) {
                if (canThrowMatchError) {
                    throw;
                }
                Cout << "--------------------------------------------------------------------------------" << Endl;
                Cout << "Result match error: " << e.what() << Endl;
                Cout << "--------------------------------------------------------------------------------" << Endl;
                return false;
            }
        }

        void AddArg(const TString& name, TArg&& arg) {
            Args_.insert(std::make_pair(name, std::move(arg)));
        }

    private:
        bool Invoke(const TTestContext& ctx, const TString& input) const {
            auto f = NCheck::TCheckFunctions::Lookup(Function_);
            return f(ctx, input, Args_);
        }

    private:
        TString Function_;

        TFunctionArgs Args_;
    };

    //
    // ResultCookie -- defines a value expected to be received in
    // Set-Cookie header of a HTTP response.
    //
    class TResultCookie {
    public:
        TResultCookie(const TString& name, std::unique_ptr<TResultItem> value)
            : Name_(name)
            , Value_(std::move(value))
        {
        }
        ~TResultCookie() = default;

        const TString& GetName() const {
            return Name_;
        }
        bool DontCare() const {
            return Value_ == nullptr;
        }
        TString GetValue() const {
            if (Value_ == nullptr) {
                throw yexception() << "Internal err: try access 'don't care cookie";
            }
            return Value_->GetValue();
        }
        bool MatchValue(const TTestContext& ctx, const TString& v) const {
            if (Value_ == nullptr) {
                throw yexception() << "Internal err: try access 'don't care cookie";
            }
            return Value_->Match(ctx, v);
        }

        // TEMP!!! Need case-insensitive map here
        using TAttribArray = std::map<TString, std::shared_ptr<TResultItem>>;
        const TAttribArray& GetAttribs() const {
            return Attribs_;
        }
        void AddAttrib(const TString& name, std::unique_ptr<TResultItem> value) {
            Attribs_[name] = std::shared_ptr<TResultItem>(value.release());
        }

        void Print(IOutputStream& s = Cout) const;

    private:
        TString Name_;
        std::unique_ptr<TResultItem> Value_;
        TAttribArray Attribs_;
    };

    //
    // Result -- define set of parameters to check after a test
    // case invocation.
    //
    class TResult {
    public:
        TResult() = default;
        ~TResult() = default;

        const TString& GetName() const {
            return Name_;
        }
        void SetName(const TString& n) {
            Name_ = n;
        }

        long GetStatusCode() const {
            return StatusCode_;
        }
        void SetStatusCode(long code) {
            StatusCode_ = code;
        }

        using TCookieArray = std::vector<std::shared_ptr<TResultCookie>>;
        const TCookieArray& GetCookies() const {
            return Cookies_;
        }
        void AddCookie(std::unique_ptr<TResultCookie> p) {
            Cookies_.push_back(std::shared_ptr<TResultCookie>(p.release()));
        }

        const TResultItem* GetXmlBody() const {
            return XmlBody_.get();
        }
        void SetXmlBody(std::unique_ptr<TResultItem> body) {
            XmlBody_ = std::move(body);
        }

        const TResultItem* GetJsonBody() const {
            return JsonBody_.get();
        }
        void SetJsonBody(std::unique_ptr<TResultItem> body) {
            JsonBody_ = std::move(body);
        }

        void Print(IOutputStream& s = Cout) const;

        bool WasUsed() const {
            return WasUsed_;
        }
        void SetUsed() const {
            WasUsed_ = true;
        }

        using THeaders = std::map<TString, std::shared_ptr<TResultItem>>;
        const THeaders& GetHeaders() const {
            return Headers_;
        }
        void AddHeader(const TString& name, std::unique_ptr<TResultItem> i) {
            Headers_.insert({name, std::shared_ptr<TResultItem>(i.release())});
        }

    private:
        TString Name_;
        long StatusCode_ = 0;

        TCookieArray Cookies_;

        std::unique_ptr<TResultItem> XmlBody_;
        std::unique_ptr<TResultItem> JsonBody_;

        THeaders Headers_;

        mutable bool WasUsed_ = false;
    };

}
