#include "scenario.h"

#include "libretto.h"
#include "pregeneration.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/string.h>

#include <memory>
#include <vector>

namespace NPassport::NLast {
    //
    // Case
    //
    TCase::TCase() = default;

    TCase::~TCase() = default;

    //
    // Scenario
    //
    TCheck::~TCheck() = default;

    TScenario::TScenario(const NLibretto::TLibretto& libretto)
        : Libretto_(libretto)
    {
        TPregeneration::GetIns().Reset();

        // Locate all node elements and proccess them
        TLog::Debug() << "Scenario: creating nodes";
        for (const NLibretto::TNode& node : Libretto_.GetNodes()) {
            Nodes_.push_back(CreateNode(node));
        }
        TLog::Debug() << "Scenario: all nodes are created";

        // For pregeneration
        if (!Pregeneration_.empty()) {
            TLog::Debug() << "Starting pregeneration";
        }
        for (const auto& func : Pregeneration_) {
            func();
        }
        // Database needs some time to send data from master to replication
        if (!TPregeneration::GetIns().IsEmpty()) {
            TLog::Debug() << "Some pregeneration happend: need to sleep for 1 second";
            sleep(1);
        }
    }

    TScenario::~TScenario() = default;

    std::vector<TString> TScenario::GetAllResults() const {
        return Libretto_.GetResultNames();
    }

    std::unique_ptr<TNode> TScenario::CreateNode(const NLibretto::TNode& xnode) {
        std::unique_ptr<TNode> node = std::make_unique<TNode>();

        node->Description(xnode.Description);
        node->SetSingleThreadOnly(xnode.SingleThreadOnly);
        node->PreTest(xnode.PreTestScript);
        node->PostTest(xnode.PostTestScript);

        // Url
        const TConfig& lastConfig = TConfig::Get();
        if (!lastConfig.Url.empty()) { // take url from cmd-option
            node->Urls({lastConfig.Url});
        } else {
            Y_ENSURE(!xnode.Url.empty());
            node->Urls(xnode.Url);
        }

        // Test cases
        TLog::Debug() << "Scenario: creating cases: " << xnode.Description;
        for (const auto& pair : xnode.Cases) {
            if (lastConfig.Cases.empty() || lastConfig.Cases.find(pair.first) != lastConfig.Cases.end()) {
                node->AddCase(CreateCase(pair.second));
            }
        }
        TLog::Debug() << "Scenario: all cases are created: " << xnode.Description;

        return node;
    }

    std::unique_ptr<TCase> TScenario::CreateCase(const NLibretto::TCase& xcase) {
        std::unique_ptr<TCase> tcase = std::make_unique<TCase>();

        tcase->Description(xcase.CheckSet.Description);
        tcase->AddVar(CreatePathRefList(xcase.CheckSet.Path));
        tcase->Redirect(xcase.Redirects);

        TLog::Debug() << "Scenario: creating checks: " << xcase.CheckSet.Description;
        for (const NLibretto::TCheck& check : xcase.Checks) {
            tcase->AddCheck(CreateCheck(check));
        }
        TLog::Debug() << "Scenario: all checks are created: " << xcase.CheckSet.Description;

        // Tvm sign
        tcase->SetNeedSignTs(xcase.TvmSignTs);
        tcase->SetNeedTvm2SignWithOldSecret(xcase.Tvm2SignOld);
        tcase->SetNeedTvm2Sign(xcase.Tvm2Sign);

        // Http
        tcase->SetPost(xcase.IsPost);
        tcase->SetCustomMethod(xcase.CustomMethod);

        // Variables
        auto add = [&tcase, this](const NLibretto::TNodeVariables& v, const auto& factory) {
            for (const NLibretto::TVariable& d : v) {
                tcase->AddVar(CreateNamedVar(factory, d, true));
            }
        };
        add(xcase.CheckSet.Cgi, TCgiNamedVarFactory());
        add(xcase.CheckSet.Cookie, TCookieNamedVarFactory());
        add(xcase.CheckSet.Header, THeaderNamedVarFactory());

        return tcase;
    }

    std::unique_ptr<TNamedVar> TScenario::CreateNamedVar(const TNamedVarFactory& factory,
                                                         const NLibretto::TVariable& xvar,
                                                         bool needCheckEmptiness) {
        std::unique_ptr<TNamedVar> var(factory.Create(xvar.Name));
        ApplyNamedVarRefList(factory.XmlName(), xvar.Name, xvar.Value, xvar.WithNull, *var);

        if (needCheckEmptiness && var->empty()) {
            throw yexception() << "'case/" << factory.XmlName()
                               << " name=\"" << xvar.Name << "\"' must have at least one value specified";
        }

        return var;
    }

    std::unique_ptr<TCheck> TScenario::CreateCheck(const NLibretto::TCheck& xcheck) {
        std::unique_ptr<TCheck> check = std::make_unique<TCheck>();

        check->Description(xcheck.CheckSet.Description);
        check->SetPath(CreatePathRefList(xcheck.CheckSet.Path));

        // Variables
        auto add = [&check, this](auto func, const NLibretto::TNodeVariables& v, const auto& factory) {
            for (const NLibretto::TVariable& d : v) {
                (check.get()->*func)(CreateNamedVar(factory, d, false));
            }
        };
        add(&TCheck::AddCgi, xcheck.CheckSet.Cgi, TCgiNamedVarFactory());
        add(&TCheck::AddCookie, xcheck.CheckSet.Cookie, TCookieNamedVarFactory());
        add(&TCheck::AddHdr, xcheck.CheckSet.Header, THeaderNamedVarFactory());

        check->SetResult(CreateResult(Libretto_.GetResult(xcheck.Result)));

        return check;
    }

    std::unique_ptr<TResult> TScenario::CreateResult(const NLibretto::TResult& xresult) {
        std::unique_ptr<TResult> result = std::make_unique<TResult>();

        result->SetName(xresult.Name);
        result->SetStatusCode(xresult.StatusCode);

        // Set-cookie
        for (const NLibretto::TSetCookie& cookies : xresult.SetCookie) {
            if (cookies.Ignore) {
                result->AddCookie(std::make_unique<TResultCookie>(cookies.Name, std::unique_ptr<TResultItem>()));
                continue;
            }

            std::unique_ptr<TResultCookie> cookie =
                std::make_unique<TResultCookie>(cookies.Name, ParseResultItem(cookies.Value));
            for (const NLibretto::TResultItem& attr : cookies.Attribute) {
                cookie->AddAttrib(attr.Name, ParseResultItem(attr));
            }
            result->AddCookie(std::move(cookie));
        }

        // Headers
        for (const NLibretto::TResultItem& h : xresult.Header) {
            result->AddHeader(h.Name, ParseResultItem(h));
        }

        // Page body
        if (xresult.XmlBody) {
            result->SetXmlBody(ParseResultItem(*xresult.XmlBody));
        }
        if (xresult.JsonBody) {
            result->SetJsonBody(ParseResultItem(*xresult.JsonBody));
        }

        return result;
    }

    std::unique_ptr<TResultItem> TScenario::ParseResultItem(const NLibretto::TResultItem& xitem) {
        if (!xitem.Func) {
            return std::make_unique<TStaticResultItem>(xitem.Value);
        }

        NCheck::TCheckFunctions::Lookup(xitem.Func); // check existence
        return std::make_unique<TDynamicResultItem>(xitem);
    }

    std::unique_ptr<TPathVar> TScenario::CreatePathRefList(const std::vector<NLibretto::TPath>& pathes) {
        if (pathes.empty()) {
            return {};
        }

        std::unique_ptr<TPathVar> res = std::make_unique<TPathVar>();

        for (const NLibretto::TPath& path : pathes) {
            if (path.WithNull) {
                res->AddValue(std::unique_ptr<TString>());
            }

            for (const TString& p : path.Values) {
                res->AddValue(std::make_unique<TString>(p));
            }
        }

        return res;
    }

    void TScenario::ApplyNamedVarRefList(
        const TString& type,
        const TString& name,
        const TString& refList,
        bool fWithNull,
        TNamedVar& var) {
        const NLibretto::TNamedVar xvar = Libretto_.GetNamedVar(type, name, refList);

        if (fWithNull) {
            var.AddValue(std::unique_ptr<TNamedVarInst>());
        }

        TArgVect sharedArgs;
        for (const NLibretto::TArg& xarg : xvar.ResultItem.Args) {
            sharedArgs.push_back(std::make_pair(xarg.Name, xarg.ExpectedValue));
        }

        for (const auto& pair : xvar.Intanses) {
            var.AddValue(
                xvar.ResultItem.Func.empty()
                    ? std::make_unique<TStaticNamedInst>(pair.first, pair.second.Value)
                    : CreateDynamicInst(pair.first, pair.second, xvar.ResultItem.Func, sharedArgs, xvar.NeedPregeneration ? name : TString()));
        }
    }

    std::unique_ptr<TNamedVarInst> TScenario::CreateDynamicInst(
        const TString& id,
        const NLibretto::TResultItem& xinst,
        const TString& fname,
        const TArgVect& sharedArgs,
        const TString& name) {
        std::unique_ptr<TDynamicNamedInst> ret = std::make_unique<TDynamicNamedInst>(id, fname);

        for (const NLibretto::TArg& xarg : xinst.Args) {
            ret->AddArg(xarg.Name, TArg(xarg));
        }

        for (const auto& pair : sharedArgs) {
            ret->AddArg(pair.first, TArg(pair.second));
        }

        if (name) {
            Pregeneration_.push_back([ptr = ret.get(), name] { ptr->Pregen(name); });
        }

        return ret;
    }
}
