#pragma once

#include "config.h"
#include "result.h"
#include "variables.h"

#include <util/generic/noncopyable.h>
#include <util/generic/string.h>
#include <util/stream/output.h>

#include <vector>

namespace NPassport::NLast::NLibretto {
    class TLibretto;
    struct TCase;
    struct TCheck;
    struct TNode;
    struct TPath;
    struct TVariable;
    struct TResult;
    struct TResultItem;
}

namespace NPassport::NLast {
    //
    // Test case - a set invocations defined as a production of
    // all defined values of the test case variables.
    //
    class TNode;
    class TCheck;
    class TCase {
    public:
        TCase();
        ~TCase();

        const TString& Description() const {
            return Description_;
        }
        void Description(const TString& d) {
            Description_ = d;
        }

        void Redirect(int r) {
            Redirect_ = r;
        }
        int Redirect() const {
            return Redirect_;
        }

        using TVarArray = std::vector<std::unique_ptr<TVariable>>;
        const TVarArray& Variables() const {
            return Variables_;
        }
        void AddVar(std::unique_ptr<TVariable> v) {
            if (v) {
                Variables_.push_back(std::move(v));
            }
        }

        using TCheckArray = std::vector<std::unique_ptr<TCheck>>;
        const TCheckArray& Checks() const {
            return Checks_;
        }
        void AddCheck(std::unique_ptr<TCheck> v) {
            Checks_.push_back(std::move(v));
        }

        void Print(IOutputStream& s) const;

        void SetNeedSignTs(bool val) {
            NeedSignTs_ = val;
        }
        bool IsNeedSignTs() const {
            return NeedSignTs_;
        }
        void SetNeedTvm2Sign(bool val) {
            NeedTvm2Sign_ = val;
        }
        bool IsNeedTvm2Sign() const {
            return NeedTvm2Sign_;
        }
        void SetNeedTvm2SignWithOldSecret(bool val) {
            NeedTvm2SignWithOldSecret_ = val;
        }
        bool IsNeedTvm2SignWithOldSecret() const {
            return NeedTvm2SignWithOldSecret_;
        }

        void SetPost(bool val) {
            IsPost_ = val;
        }
        bool IsPost() const {
            return IsPost_;
        }

        void SetCustomMethod(const TString& customRequest) {
            CustomMethod_ = customRequest;
        }
        const TString& CustomMethod() const {
            return CustomMethod_;
        }

    private:
        TString Description_;
        TVarArray Variables_;
        int Redirect_ = 0;

        TCheckArray Checks_;
        bool NeedSignTs_ = false;
        bool NeedTvm2Sign_ = false;
        bool NeedTvm2SignWithOldSecret_ = false;
        bool IsPost_ = false;
        TString CustomMethod_;
    };

    //
    // Tested node
    //
    class TNode {
    public:
        void Urls(const std::vector<TString>& urls) {
            Urls_ = urls;
        }
        const std::vector<TString>& Urls() const {
            return Urls_;
        }

        TString Description() const {
            return Description_;
        }
        void Description(const TString& description) {
            Description_ = description;
        }

        TString PreTest() const {
            return PreTest_;
        }
        void PreTest(const TString& pre_test) {
            PreTest_ = pre_test;
        }

        TString PostTest() const {
            return PostTest_;
        }
        void PostTest(const TString& post_test) {
            PostTest_ = post_test;
        }

        using TCases = std::vector<std::unique_ptr<TCase>>;
        const TCases& Cases() const {
            return Cases_;
        }

        void AddCase(std::unique_ptr<TCase> c) {
            Cases_.push_back(std::move(c));
        }

        void Print(IOutputStream& s = Cout) const;

        void SetSingleThreadOnly(bool f) {
            AllowSingleThreadOnly_ = f;
        }

        bool AllowSingleThreadOnly() const {
            return AllowSingleThreadOnly_;
        }

    private:
        TCases Cases_;
        TString Description_;
        TString PreTest_;
        TString PostTest_;
        std::vector<TString> Urls_;
        bool AllowSingleThreadOnly_ = false;
    };

    //
    // Check -- associates result to test against after a case invocation
    // with a subset of a test case.
    //
    class TPathVar;
    class TNamedVar;
    class TCheck {
    public:
        TCheck() = default;
        ~TCheck();

        const TPathVar* GetPath() const {
            return Path_.get();
        }
        void SetPath(std::unique_ptr<TPathVar> p) {
            Path_ = std::move(p);
        }

        TString Description() const {
            return Description_;
        }
        void Description(const TString& description) {
            Description_ = description;
        }

        using TValues = std::vector<std::unique_ptr<TNamedVar>>;

        const TValues& Cgi() const {
            return Cgi_;
        }
        void AddCgi(std::unique_ptr<TNamedVar> v) {
            Cgi_.push_back(std::move(v));
        }

        const TValues& Hdr() const {
            return Header_;
        }
        void AddHdr(std::unique_ptr<TNamedVar> v) {
            Header_.push_back(std::move(v));
        }

        const TValues& Cookie() const {
            return Cookie_;
        }
        void AddCookie(std::unique_ptr<TNamedVar> v) {
            Cookie_.push_back(std::move(v));
        }

        void SetResult(std::unique_ptr<TResult> r) {
            Result_ = std::move(r);
        }
        const TResult* GetResult() const {
            return Result_.get();
        }

        void Print(IOutputStream& s = Cout) const;

    private:
        std::unique_ptr<TPathVar> Path_;
        TString Description_;

        TValues Cgi_;
        TValues Header_;
        TValues Cookie_;

        std::unique_ptr<TResult> Result_;
    };

    //
    // Helper std::exception classes
    //
    class TSingleChildError: public yexception {
    public:
        TSingleChildError(const TStringBuf parent, const TStringBuf child) {
            (*this) << "Element '" << parent << "' must have exactly one '" << child << "' child";
        }
    };

    class TSingleElementError: public yexception {
    public:
        TSingleElementError(const TStringBuf xpath) {
            (*this) << "There must be exactly one item in the set '" << xpath << "'";
        }
    };

    //
    // Entire test scenario
    //
    class TNamedVarFactory;
    class TScenario {
    public:
        TScenario(const NLibretto::TLibretto& libretto);
        ~TScenario();

        using TNodes = std::vector<std::unique_ptr<TNode>>;
        const TNodes& Nodes() const {
            return Nodes_;
        }

        void Print(IOutputStream& s = Cout) const;

        std::vector<TString> GetAllResults() const;

    private:
        using TArgVect = std::vector<std::pair<TString, TString>>;

        //
        // Impl routines
        //
        std::unique_ptr<TNode> CreateNode(const NLibretto::TNode& xnode);
        std::unique_ptr<TCase> CreateCase(const NLibretto::TCase& xcase);
        std::unique_ptr<TNamedVar> CreateNamedVar(const TNamedVarFactory& factory,
                                                  const NLibretto::TVariable& xvar,
                                                  bool needCheckEmptiness);
        std::unique_ptr<TCheck> CreateCheck(const NLibretto::TCheck& xcheck);
        static std::unique_ptr<TResult> CreateResult(const NLibretto::TResult& xresult);
        static std::unique_ptr<TResultItem> ParseResultItem(const NLibretto::TResultItem& xitem);
        static std::unique_ptr<TPathVar> CreatePathRefList(const std::vector<NLibretto::TPath>& pathes);
        void ApplyNamedVarRefList(
            const TString& type,
            const TString& name,
            const TString& refList,
            bool fWithNull,
            TNamedVar& var);

        std::unique_ptr<TNamedVarInst> CreateDynamicInst(const TString& id,
                                                         const NLibretto::TResultItem& xinst,
                                                         const TString& fname,
                                                         const TArgVect& shared,
                                                         const TString& name);

    private:
        //
        // Data
        //
        const NLibretto::TLibretto& Libretto_;

        TNodes Nodes_;
        std::vector<std::function<void()>> Pregeneration_;
    };

    //
    // Helper classes, holding meta-info about specific NamedVariable subclasses
    //
    class TNamedVarFactory {
    public:
        virtual const char* XmlName() const = 0;
        virtual std::unique_ptr<TNamedVar> Create(const TString& name) const = 0;
    };

    class TCgiNamedVarFactory: public TNamedVarFactory {
    public:
        const char* XmlName() const override {
            return "cgi";
        }
        std::unique_ptr<TNamedVar> Create(const TString& name) const override {
            return std::make_unique<TCgiVar>(name);
        }
    };

    class TCookieNamedVarFactory: public TNamedVarFactory {
    public:
        const char* XmlName() const override {
            return "cookie";
        }
        std::unique_ptr<TNamedVar> Create(const TString& name) const override {
            return std::make_unique<TCookieVar>(name);
        }
    };

    class THeaderNamedVarFactory: public TNamedVarFactory {
    public:
        const char* XmlName() const override {
            return "header";
        }
        std::unique_ptr<TNamedVar> Create(const TString& name) const override {
            return std::make_unique<THeaderVar>(name);
        }
    };

}
