#pragma once

#include <contrib/libs/curl/include/curl/curl.h>

#include <util/generic/string.h>
#include <util/generic/yexception.h>

#include <vector>

namespace NCurlwrap {
    class TCurlError: public yexception {
    };

    class TSession {
    public:
        TSession() {
            CURLcode ret = curl_global_init(CURL_GLOBAL_ALL);
            if (ret != CURLE_OK) {
                throw TCurlError() << "curl_global_init() failed";
            }
        }

        ~TSession() {
            curl_global_cleanup();
        }
    };

    class THandle {
    public:
        THandle() {
            Handle_ = curl_easy_init();
            if (Handle_ == nullptr) {
                throw TCurlError() << "curl_easy_init() failed";
            }

            // Must do this before calling any other own method
            ErrorText_[0] = '\0';
            CURLcode ret = curl_easy_setopt(Handle_, CURLOPT_ERRORBUFFER, ErrorText_);
            if (ret != CURLE_OK) {
                throw TCurlError() << "curl_easy_setopt(CURLOPT_ERRORBUFFER) failed";
            }
        }

        ~THandle() {
            curl_easy_cleanup(Handle_);
        }

        void Perform() {
            CURLcode ret = curl_easy_perform(Handle_);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        template <class T>
        void Set(CURLoption option, T value) {
            CURLcode ret = curl_easy_setopt(Handle_, option, (long)value);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        void Set(CURLoption option, const TString& value) {
            const char* s = MemOpts_.Add(option, value);
            CURLcode ret = curl_easy_setopt(Handle_, option, s);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        void Set(CURLoption option, const char* value) {
            Set(option, TString(value));
        }

        void Set(CURLoption option, const void* value) {
            CURLcode ret = curl_easy_setopt(Handle_, option, value);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        void Set(curl_debug_callback cb) {
            CURLcode ret = curl_easy_setopt(Handle_, CURLOPT_DEBUGFUNCTION, cb);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        void Set(curl_write_callback cb) {
            CURLcode ret = curl_easy_setopt(Handle_, CURLOPT_WRITEFUNCTION, cb);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        using TCurlHeaderCallback = size_t (*)(void* ptr, size_t size, size_t nmemb, void* stream);
        void Set(TCurlHeaderCallback cb) {
            CURLcode ret = curl_easy_setopt(Handle_, CURLOPT_HEADERFUNCTION, cb);
            if (ret != CURLE_OK) {
                throw TCurlError() << ErrorText_;
            }
        }

        void GetInfo(CURLINFO info, long* out) {
            CURLcode ret = curl_easy_getinfo(Handle_, info, out);
            if (ret != CURLE_OK) {
                throw TCurlError() << "Err code=" << (int)ret << "('" << ErrorText_ << "')";
            }
        }

        THandle(const THandle& h) {
            Clone(h);
        }
        THandle& operator=(const THandle& h) {
            Clone(h);
            return *this;
        }

    private:
        void Clone(const THandle& h) {
            // Duplicate CURL handle. Use source handle's error buffer.
            Handle_ = curl_easy_duphandle(h.Handle_);
            if (Handle_ == nullptr) {
                throw TCurlError() << h.ErrorText_;
            }

            // Set up error buffer.
            // Must do this before calling any other CURL function on this handle.
            CURLcode ret = curl_easy_setopt(Handle_, CURLOPT_ERRORBUFFER, ErrorText_);
            if (ret != CURLE_OK) {
                throw TCurlError() << "curl_easy_setopt(CURLOPT_ERRORBUFFER) failed";
            }

            // Copy options set
            for (TMemOpts::TItems::const_iterator it = h.MemOpts_.begin(); it != h.MemOpts_.end(); ++it) {
                Set(it->first, it->second);
            }
        }

        class TMemOpts {
        public:
            TMemOpts() {
                V_.reserve(10);
            }

            ~TMemOpts() {
                for (size_t i = 0; i < V_.size(); ++i) {
                    delete[] V_[i].second;
                }
            }

            /*
        MemOpts &operator= (const MemOpts &another)
        {
            v_.resize (another.v_.size ());
            for (Items::const_iterator it = another.v_.begin (); it != another.v_.end (); ++it) {
                char    *p;
                if (it->second != nullptr) {
                    p = strdup (it->second);
                    if (p == nullptr)
                        throw CurlError ("Wrapper out of memory");
                } else p = nullptr;
                v_.push_back (Item (it->first, p));
            }

            return *this;
        }
*/

            const char* Add(CURLoption option, const TString& value) {
                char* p = new char[value.size() + 1];
                std::strcpy(p, value.c_str());
                V_.resize(V_.size() + 1);
                V_.back().first = option;
                V_.back().second = p;
                return p;
            }

            using TItem = std::pair<CURLoption, char*>;
            using TItems = std::vector<TItem>;
            TItems::const_iterator begin() const {
                return V_.begin();
            }
            TItems::const_iterator end() const {
                return V_.end();
            }

            TMemOpts(const TMemOpts&) = delete;
            TMemOpts& operator=(const TMemOpts& another) = delete;

        private:
            TItems V_;
        };

        CURL* Handle_ = nullptr;
        TMemOpts MemOpts_;
        char ErrorText_[CURL_ERROR_SIZE]{};
    };

    class TSlist {
    public:
        TSlist() = default;

        ~TSlist() {
            curl_slist_free_all(Ptr_);
        }

        void Append(const TString& s) {
            Ptr_ = curl_slist_append(Ptr_, s.c_str());
        }

        const struct curl_slist* Ptr() const {
            return Ptr_;
        }

    private:
        struct curl_slist* Ptr_ = nullptr;
    };

}
