#include "session_utils.h"

#include <passport/infra/tools/ylast/src/config.h>
#include <passport/infra/tools/ylast/src/error.h>
#include <passport/infra/tools/ylast/src/context/sess_context.h>

#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <algorithm>
#include <time.h>

namespace NPassport::NLast {
    //
    // Time-handling helpers
    //
    time_t CalcTimeFromDelta(const TString& strdelta) {
        char* end;
        long delta = strtol(strdelta.c_str(), &end, 10);
        if (*end != '\0') {
            throw TLastError() << "CalcTimeFromDelta(): invalid 'time=" << strdelta << "' argument";
        }

        // Determine current time
        TInstant now = TInstant::Now();
        if (now.MicroSecondsOfSecond() > 500000L) {
            now += TDuration::Seconds(1);
        }

        // Return current time adjusted by the specified time delta
        return now.Seconds() + delta;
    }

    time_t CalcTimeDiffFromDelta(const TString& strdelta, const TString& strbase) {
        char* end;
        long delta = strtol(strdelta.c_str(), &end, 10);
        if (*end != '\0') {
            throw TLastError() << "CalcTimeDiffFromDelta(): invalid 'time=" << strdelta << "' argument";
        }
        long base = strtol(strbase.c_str(), &end, 10);
        if (*end != '\0') {
            throw TLastError() << "CalcTimeDiffFromDelta(): invalid 'time=" << strbase << "' argument";
        }

        // Determine current time
        TInstant now = TInstant::Now();
        if (now.MicroSecondsOfSecond() > 500000L) {
            now += TDuration::Seconds(1);
        }

        // Return current time adjusted by the specified time delta
        return now.Seconds() + delta - base;
    }

    TString FormatTimeFromDelta(const TString& strdelta) {
        return IntToString<10>(CalcTimeFromDelta(strdelta));
    }

    TUserSection::TUserSection(const TString& uid)
        : Uid(uid)
        , PwdCheckTime("-1")
        , Flags("0")
    {
    }

    const char* const TSessionContext::statusStrings_[] = {
        "VALID",
        "NEED_RESET",
        "EXPIRED",
        "SIGN_BROKEN",
        "NO_COOKIE",
        "CANT_CHECK",
        "NOAUTH",
        "DISABLED"};

    const size_t TSessionContext::statusCount_ = sizeof(TSessionContext::statusStrings_) / sizeof(TSessionContext::statusStrings_[0]);

    const TSessionContext TSessionContext::default_;
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TSessionContext::TKeyMap TSessionContext::domainKeys_;
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TSessionContext::TKeyspaceIdMap TSessionContext::domainIds_;

    TSessionContext::TKeyData::TKeyData(const TString& id, const TString& body)
        : Id(id)
    {
        Binbody = NUtils::Hex2bin(body);
        Hash = NUtils::TCrypto::Sha256(Binbody);
    }

    TSessionContext::TSessionContext()
        : InvAge("0")
        , Host("beta.yandex.ua")
        , Ttl("2")
        , Uid("100001")
        , AuthidTime("1234567890987")
        , AuthidIp("AQAAfw")
        , AuthidId("fe")
        , Dompref("")
        , Domsuff("yandex_ua")
        , Keynum("10046")
        , Rnd("9383")
    {
    }

    TSessionContext::TSessionContext(const TFunctionArgs& args) {
        *this = default_;

        TFunctionArgs::const_iterator it;

        // Locate cookie version
        it = args.find("version");
        if (it != args.end()) {
            if (it->second.Value() != "3") {
                throw TLastError() << "Unsupported Session_id or session_info version: " << it->second.Value();
            }
        }

        Version = 3;

        // Locate 'time' argument
        it = args.find("time");
        if (it != args.end()) {
            InvAge = it->second;
        }

        // Locate 'host' argument
        it = args.find("host");
        if (it == args.end()) {
            throw TLastError() << "Session_id or session_info or syn_info context: no 'host' argument";
        }
        Host = it->second;

        // Determine key for this domain
        TString dompref;
        TKeyMap::iterator domit = HostToSuffix(Host, dompref);
        Domsuff = domit->first;
        Keynum = domit->second.Id;
        Keybinbody = domit->second.Binbody;
        Keybodyhash = domit->second.Hash;
        if (args.find("addpref") != args.end()) {
            Dompref = std::move(dompref);
        }

        // Locate 'flag' argument
        it = args.find("flag");
        if (it != args.end()) {
            if (it->second.Value() == "session") {
                Ttl = "0";
            } else if (it->second.Value() == "permanent") {
                Ttl = "1";
            } else if (it->second.Value() == "regular") {
                Ttl = "2";
            } else if (it->second.Value() == "2weeks") {
                Ttl = "3";
            } else if (it->second.Value() == "restricted") {
                Ttl = "4";
            } else if (it->second.Value() == "permanentnew") {
                Ttl = "5";
            } else {
                throw TLastError()
                    << "Session_id or session_info or syn_info context: unknown 'flag' value: "
                    << it->second;
            }
        }

        // Locate 'uid' argument
        it = args.find("uid");
        if (it != args.end()) {
            Uid = it->second;
        }

        // Locate sid 8 extensions
        it = args.find("authtime");
        if (it != args.end()) {
            AuthidTime = it->second;
        } else {
            it = args.find("authdelta");
            if (it != args.end()) {
                AuthidTime = FormatTimeFromDelta(it->second) + "111";
            }
        }

        it = args.find("authip");
        if (it != args.end()) {
            NUtils::TIpAddr addr;

            if (!addr.Parse(it->second.Value())) {
                throw TLastError() << "Invalid authip format";
            }

            AuthidIp = addr.ToBase64String();
        }

        it = args.find("authid");
        if (it != args.end()) {
            AuthidId = it->second;
        }

        // v2 extensions
        it = args.find("safe");
        if (args.end() != it) {
            Safe = it->second;
        }

        // v3 extensions
        it = args.find("default_index");
        DefIdx = (it != args.end()) ? it->second.Value() : "0";

        it = args.find("cookie_flags");
        Flags = (it != args.end()) ? it->second.Value() : "1";

        // update v3 cookie flags value if safe flag is on
        if (!Safe.empty()) {
            char* end;
            unsigned val = strtol(Flags.c_str(), &end, 16);
            if (*end == 0) {       // parsed int
                if (Safe == "0") { // clear safe flag
                    val &= 0xfffffffe;
                } else { // set safe flag
                    val |= 1;
                }
            }
            Flags = IntToString<10>(val);
        }

        it = args.find("cookie_kv_ext");
        if (args.end() != it) {
            KvExt = it->second;
        }

        it = args.find("login_id");
        if (it != args.end()) {
            if (!KvExt.empty()) {
                KvExt.push_back('.');
            }

            KvExt.append("1:").append(NUtils::Bin2base64url(it->second.Value()));
        }

        // ext_userip, ext_auth_ts, ext_update_ts
        it = args.find("ext_userip");
        if (it != args.end()) {
            NUtils::TIpAddr addr;

            if (!addr.Parse(it->second)) {
                throw TLastError() << "Invalid ext_userip format";
            }

            if (!KvExt.empty()) {
                KvExt.push_back('.');
            }

            KvExt.append("100:").append(addr.ToBase64String());
        }

        it = args.find("ext_auth_time");
        if (it != args.end()) {
            if (!KvExt.empty()) {
                KvExt.push_back('.');
            }

            KvExt.append("101:").append(it->second);
        } else {
            it = args.find("ext_auth_delta");
            if (it != args.end()) {
                if (!KvExt.empty()) {
                    KvExt.push_back('.');
                }

                KvExt.append("101:").append(FormatTimeFromDelta(it->second));
            }
        }

        it = args.find("ext_update_time");
        if (it != args.end()) {
            if (!KvExt.empty()) {
                KvExt.push_back('.');
            }

            KvExt.append("102:").append(it->second);
        } else {
            it = args.find("ext_update_delta");
            if (it != args.end()) {
                if (!KvExt.empty()) {
                    KvExt.push_back('.');
                }

                KvExt.append("102:").append(FormatTimeFromDelta(it->second));
            }
        }

        // allow max 5 users in test cookie
        TString suffixes[] = {"", "_1", "_2", "_3", "_4", "_5", "_6", "_7", "_8", "_9"};

        for (unsigned idx = 0; idx < sizeof(suffixes) / sizeof(TString); ++idx) {
            it = args.find("uid" + suffixes[idx]);
            if (it == args.end()) { // no more uids
                break;
            }
            TUserSection user(it->second);

            it = args.find("pwdtime" + suffixes[idx]);
            user.PwdCheckTime = (it != args.end()) ? it->second.Value() : "-1";

            it = args.find("flags" + suffixes[idx]);
            user.Flags = (it != args.end()) ? it->second.Value() : "0";

            it = args.find("lang" + suffixes[idx]);
            if (it != args.end()) {
                user.Lang = it->second;
            }

            it = args.find("socprofile" + suffixes[idx]);
            if (it != args.end()) {
                user.SocProfile = it->second;
            }

            it = args.find("login_time" + suffixes[idx]);
            if (it != args.end()) {
                user.LoginTimeDiff = it->second;
            }

            it = args.find("kv_ext" + suffixes[idx]);
            if (it != args.end()) {
                user.KvExt = it->second;
            }

            Users.push_back(user);
        }

        // TODO: drop it when all sessions will be gammed: PASSP-31300
        it = args.find("use_gamma");
        if (it != args.end()) {
            UseGamma = NUtils::ToBoolean(it->second);
        }
    }

    void TSessionContext::SetupDomainKeys(NDbPool::TDbPool& dbp, NAuth::TSessionSigner& sessSigner) {
        auto dbKeyspaces = NDbPool::NUtils::DoQueryTries(dbp, "select domainsuff, tablename, groupid from keyspaces where inuse<>0", 3).Result;

        TString domain;
        TString table;
        TString domain_id;
        while (dbKeyspaces->Fetch(domain, table, domain_id)) {
            TString selExpr("select id, keybody from ");
            selExpr.append(table).append(" order by start limit 1");
            auto dbRes = NDbPool::NUtils::DoQueryTries(dbp, selExpr, 3).Result;

            TString id;
            TString body;
            while (dbRes->Fetch(id, body)) {
                domainKeys_.insert(std::make_pair(domain, TKeyData(id, body)));
                domainIds_.insert(std::make_pair(domain, domain_id));
                std::replace(domain.begin(), domain.vend(), '_', '.');
                sessSigner.AddKeyspace(domain);
            }
        }
    }

    TSessionContext::TKeyMap::iterator TSessionContext::HostToSuffix(const TString& host, TString& dompref) {
        TString domain(host);

        try {
            std::vector<TString::size_type> dots;
            TString::size_type dotpos = 0;
            while ((dotpos = domain.find('.', dotpos)) != TString::npos) {
                if (dotpos == domain.size() - 1) {
                    throw false;
                }
                domain[dotpos] = '_';
                dots.push_back(dotpos++);
            }

            dompref.clear();
            TKeyMap::iterator domit = domainKeys_.find(domain);
            if (domit != domainKeys_.end()) {
                return domit;
            }

            for (size_t i = 0; i < dots.size(); ++i) {
                if (domit != domainKeys_.end()) {
                    return domit;
                }
                domit = domainKeys_.find(domain.substr(dots[i] + 1));
                dompref.assign(domain, 0, dots[i]);
            }
            throw TLastError() << "Scenario error: unknown domain for host '" << domain << "'";

        } catch (bool e) {
            throw TLastError() << "Scenario error: invalid domain '" << domain << "'";
        }
    }

    long TSessionContext::Age() const {
        char* end;
        long delta = strtol(InvAge.c_str(), &end, 10);
        if (*end != '\0') {
            throw TLastError() << "Session_id or session_info context: bad 'time' format";
        }
        return -delta;
    }

    TString TSessionContext::ComposeSession(const TString& defaultAge) const {
        switch (Version) {
            case 3:
                return ComposeSessionV3(defaultAge);
            default:
                throw TLastError() << "Unsupported session version requested";
        }
    }

    TString TSessionContext::ComposeSessionV3(const TString& defaultAge) const {
        TString res;
        TStringStream straux;

        TString age;
        if (!defaultAge.empty()) {
            age = InvAge.empty() || InvAge == "0" ? defaultAge : InvAge;
        } else {
            age = InvAge;
        }

        straux << "3:" << CalcTimeFromDelta(age) << '.' << Ttl << '.' << DefIdx << '.';
        straux << AuthidTime << ':' << AuthidIp << ':' << AuthidId << '.' << Flags;
        if (!KvExt.empty()) {
            straux << '.' << KvExt;
        }

        for (const TUserSection& user : Users) {
            straux << '|' << user.Uid << '.' << user.PwdCheckTime << '.' << user.Flags;
            if (!user.Lang.empty()) {
                straux << ".0:" << user.Lang;
            }
            if (!user.SocProfile.empty()) {
                straux << ".1:" << user.SocProfile;
            }
            if (!user.LoginTimeDiff.empty()) {
                TString timestamp = AuthidTime.substr(0, AuthidTime.size() - 3);
                straux << ".2:" << CalcTimeDiffFromDelta(user.LoginTimeDiff, timestamp);
            }
            if (!user.KvExt.empty()) {
                straux << '.' << user.KvExt;
            }
        }

        straux << '|';
        TString domid = domainIds_[Domsuff];

        if (!Dompref.empty()) {
            straux << Dompref << ':' << domid << ':';
        } else if (!Domsuff.empty()) {
            straux << domid << ':';
        }

        NAuth::TCombinedKeyId keyid(IntFromString<ui32, 10>(Keynum));
        TString keybody = Keybinbody;

        const NAuth::TGammaKeeperSettings& settings = TConfig::Get().GammaKeeperSettings;
        if (UseGamma) {
            auto itKeyspace = settings.KeyspaceToType.find(Domsuff);
            if (itKeyspace != settings.KeyspaceToType.end() &&
                itKeyspace->second == NAuth::EEnitityType::Session) {
                keyid = NAuth::TCombinedKeyId(settings.SigningGamma, keyid.AsNumber());

                auto it = settings.Gammas.find(settings.SigningGamma);
                Y_ENSURE(it != settings.Gammas.end(),
                         "missing gammaid " << settings.SigningGamma);
                const TString actualGamma = NUtils::TCrypto::Sha256(
                    NUtils::CreateStr("some const string", (TStringBuf)*it->second.Body));
                keybody = NAuth::TKeyWithGamma::MakeXor(actualGamma, Keybodyhash);
            }
        }

        straux << keyid.AsNumber() << '.' << Rnd;
        res.assign(straux.Str());
        res.append(".").append(NUtils::Bin2base64url(NUtils::TCrypto::HmacSha1(keybody, straux.Str())));

        return res;
    }

    TString TIdComposer::NoAuth() const {
        // Locate noauth 'time' argument (which is delta-from-now, in sec)
        // and convert it to numeric
        TFunctionArgs::const_iterator it = Args_.find("time");
        if (it == Args_.end()) {
            throw TLastError() << "IdComposer::NoAuth(): no 'time' argument";
        }

        return NUtils::CreateStr("noauth/", CalcTimeFromDelta(it->second));
    }

    TString TIdComposer::SignBroken() const {
        TString s = Context_.ComposeSession();

        // Invalidate sign by cutting one character off of the tail
        s = s.substr(0, s.size() - 1);
        Validate(s, NAuth::TSession::SIGN_BROKEN);

        return s;
    }

    TString TIdComposer::Disabled() const {
        TString s = Context_.ComposeSession();

        // Dont validate this cookie since the check will fail
        // Validate (s, auth::Session::DISABLED);

        return s;
    }

    TString TIdComposer::Valid() const {
        TString s = Context_.ComposeSession();
        Validate(s, NAuth::TSession::VALID);

        return s;
    }

    TString TIdComposer::NeedReset() const {
        TString s = Context_.ComposeSession(Context_.Ttl != "1" ? "-3601" : "-1382400");
        Validate(s, NAuth::TSession::NEED_RESET);

        return s;
    }

    TString TIdComposer::Expired() const {
        TString s = Context_.ComposeSession(Context_.Ttl != "1" ? "-7500" : "-1382400");
        Validate(s, NAuth::TSession::EXPIRED);

        return s;
    }

    TString TIdComposer::NoCookie() const {
        // The value below is too short and contains to fee parts to
        // return anything but NO_COOKIE
        TString s("a.b.c");
        Validate(s, NAuth::TSession::NO_COOKIE);

        return s;
    }

    TString TIdComposer::CantCheck() {
        // TEMP!!!
        throw TLastError() << "Internal error: CANT_CHECK Session_id generation is not supported.";
    }

    void TIdComposer::Validate(const TString& value, NAuth::TSession::EStatus expected) const {
        // TEMP!!! Shall determine host from the context_.domsuffix
        NAuth::TSession s(TSessContext::GetInstance().Parser().ParseCookie(value, Context_.Host));
        if (s.IsValid() != expected) {
            throw TLastError()
                << "Internal error: failed to generate Session_id cookie: "
                << value << ": "
                << "expected " << TSessionContext::StatusString(expected)
                << ", got " << TSessionContext::StatusString(s.IsValid());
        }
    }
}
