#include "variables.h"

#include "pregeneration.h"
#include "scenario.h"
#include "context/ticket_context.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/tvmauth/src/utils.h>

#include <util/generic/string.h>

#include <algorithm>
#include <ctime>
#include <memory>

namespace NPassport::NLast {
    //
    // Variables implementation
    //
    std::unique_ptr<TVariator> TCgiVar::GetVariator() const {
        return std::make_unique<TCgiVariator>(Name_, Values_);
    }

    std::unique_ptr<TVariator> THeaderVar::GetVariator() const {
        return std::make_unique<THeaderVariator>(Name_, Values_);
    }

    std::unique_ptr<TVariator> TCookieVar::GetVariator() const {
        return std::make_unique<TCookieVariator>(Name_, Values_);
    }

    std::unique_ptr<TVariator> TPathVar::GetVariator() const {
        return std::make_unique<TPathVariator>(Values_);
    }

    //
    // DynamicNamedInst
    //
    TString TDynamicNamedInst::GetValue() const {
        if (IsPregen_) {
            return TPregeneration::GetIns().GetValue(CgiName_, Id_);
        }

        auto f = NGen::TGenerateFunctions::Lookup(Function_);
        return f(Args_);
    }

    void TDynamicNamedInst::Pregen(const TString& name) {
        if (!name) {
            return;
        }

        if (!TPregeneration::GetIns().HasValue(name, Id_)) {
            TPregeneration::GetIns().AddValue(name, Id_, GetValue());
        }

        IsPregen_ = true;
        CgiName_ = name;
    }

    inline bool BeginsWith(const TString& full, const TString& prefix) {
        return full.compare(0, prefix.size(), prefix) == 0;
    }

    //
    // VarSet implementation
    //
    TVarSet::TVarSet(const TString& url, const TCase& cs)
        : Case_(cs)
    {
        for (const auto& var : Case_.Variables()) {
            Variators_.push_back(var->GetVariator());
        }
        if (BeginsWith(url, "http://")) {
            Schema_ = "http://";
        } else if (BeginsWith(url, "https://")) {
            Schema_ = "https://";
        }
    }

    TVarSet::~TVarSet() = default;

    bool TVarSet::Next(TTestContext& ctx) {
        // Any more combinations?
        if (Complete_) {
            return false;
        }

        // Permutate
        Complete_ = true;
        for (const auto& var : Variators_) {
            if (var->Next()) {
                Complete_ = false;
                break;
            }
        }

        // Setup expected result (if any)
        const TCheck* check = nullptr;
        for (const auto& ch : Case_.Checks()) {
            if (Match(*ch)) {
                check = ch.get();
                break;
            }
        }
        if (check == nullptr) {
            Cout << Endl;
            Print();
            throw yexception() << "No result defined for the variables combination above. Bailing out.";
        }

        // Apply
        ctx.Clear();
        for (const auto& var : Variators_) {
            var->AddSelf(ctx);
        }
        ctx.Schema = Schema_;
        ctx.Check = check;

        ctx.IsNeedSignTs = Case_.IsNeedSignTs();
        ctx.IsNeedTvm2Sign = Case_.IsNeedTvm2Sign();
        ctx.IsNeedTvm2SignWithOldSecret = Case_.IsNeedTvm2SignWithOldSecret();

        return true;
    }

    size_t TVarSet::Count() const {
        if (Variators_.empty()) {
            return 0;
        }

        size_t res = 1;
        for (const auto& var : Variators_) {
            res *= var->Count();
        }
        return res;
    }

    const TString& TSigner::GetArg(TTestContext& ctx, const TString& arg, bool mandatory = true) {
        auto it = ctx.Cgis.find(arg);
        if (it == ctx.Cgis.end()) {
            if (mandatory) {
                throw yexception() << "Need sign for tvm but no " << arg << " in request";
            }
            static const TString EMPTY_;
            return EMPTY_;
        }
        return it->second;
    }

    void TSigner::SignTsRequest(TTestContext& ctx) {
        static const TString DEPRECATED_CLIENT_ID_ = "deprecated_client_id";
        static const TString CLIENT_ID_ = "client_id";
        static const TString TS_ = "ts";
        static const TString DEPRECATED_TS_SIGN_ = "deprecated_ts_sign";
        static const TString TS_SIGN_ = "ts_sign";

        TString ts = IntToString<10>(std::time(nullptr));
        ctx.Cgis.insert(std::make_pair(TS_, ts));

        TString clId;
        TString param;
        auto it = ctx.Cgis.find(DEPRECATED_CLIENT_ID_);
        if (ctx.Cgis.end() == it) {
            param = TS_SIGN_;
            it = ctx.Cgis.find(CLIENT_ID_);
            if (ctx.Cgis.end() != it) {
                clId = it->second;
            }
        } else {
            param = DEPRECATED_TS_SIGN_;
            clId = it->second;
        }

        const TString& secret = TTicketContext::GetInstance().GetSecret(clId);
        ctx.Cgis.insert(std::make_pair(param, NUtils::Bin2base64url(NUtils::TCrypto::HmacSha256(secret, ts))));
    }

    void TSigner::SignTvm2Request(TTestContext& ctx) {
        static const TString SRC_ = "src";
        static const TString DST_ = "dst";
        static const TString SCOPES_ = "scopes";
        static const TString TS_ = "ts";
        static const TString SIGN_ = "sign";

        TString ts = IntToString<10>(std::time(nullptr));
        ctx.Cgis.insert(std::make_pair(TS_, ts));
        const TString& src = GetArg(ctx, SRC_);
        const TString& dst = GetArg(ctx, DST_);
        const TString& scopes = GetArg(ctx, SCOPES_, false);
        ctx.Cgis.insert(std::make_pair(
            SIGN_,
            NTvmAuth::NUtils::SignCgiParamsForTvm(ctx.IsNeedTvm2SignWithOldSecret
                                                      ? TTicketContext::GetInstance().GetOldSecret(src)
                                                      : TTicketContext::GetInstance().GetSecret(src),
                                                  ts,
                                                  dst,
                                                  scopes)));
    }

    bool TVarSet::Match(const TCheck& check) {
        // To prevent same variator to match more than one statement
        // in a check definition mark every matched variator. So in
        // the begining we need to clear all marks.
        for (const auto& v : Variators_) {
            v->MarkUnmatched();
        }

        // Try match path
        if (check.GetPath() != nullptr) {
            auto v = std::find_if(Variators_.begin(),
                                  Variators_.end(),
                                  [&check](const std::unique_ptr<TVariator>& v) -> bool {
                                      return v->MatchPath(check.GetPath());
                                  });
            if (v == Variators_.end()) {
                return false;
            }
            (*v)->MarkMatched();
        }

        // Try match CGI variables
        for (const auto& val : check.Cgi()) {
            auto v = std::find_if(Variators_.begin(),
                                  Variators_.end(),
                                  [&val](const std::unique_ptr<TVariator>& v) -> bool {
                                      return v->MatchCgi(*val);
                                  });
            if (v == Variators_.end()) {
                return false;
            }
            (*v)->MarkMatched();
        }

        // Try match Header variables
        for (const auto& val : check.Hdr()) {
            auto v = std::find_if(Variators_.begin(),
                                  Variators_.end(),
                                  [&val](const std::unique_ptr<TVariator>& v) -> bool {
                                      return v->MatchHeader(*val);
                                  });
            if (v == Variators_.end()) {
                return false;
            }
            (*v)->MarkMatched();
        }

        // Try match Cookie variables
        for (const auto& val : check.Cookie()) {
            auto v = std::find_if(Variators_.begin(),
                                  Variators_.end(),
                                  [&val](const std::unique_ptr<TVariator>& v) -> bool {
                                      return v->MatchCookie(*val);
                                  });
            if (v == Variators_.end()) {
                return false;
            }
            (*v)->MarkMatched();
        }

        return true;
    }

    void TSigner::Sign(TTestContext& ctx) {
        if (!ctx.IsNeedSignTs && !ctx.IsNeedTvm2Sign && !ctx.IsNeedTvm2SignWithOldSecret) {
            return;
        }

        TTicketContext::GetInstance();
        try {
            if (ctx.IsNeedTvm2Sign || ctx.IsNeedTvm2SignWithOldSecret) {
                TSigner::SignTvm2Request(ctx);
            }

            if (ctx.IsNeedSignTs) {
                TSigner::SignTsRequest(ctx);
            }
        } catch (const std::exception& e) {
            // For cases where it is impossible to create correct sign, we create 'something'
            // only to ensure arg 'env_sign' exists
            Cerr << "    No correct sign: " << e.what() << Endl;
            ctx.Cgis.insert(std::make_pair("env_sign", "123"));
        }
    }

}
