#pragma once

#include "func.h"
#include "result.h"

#include <util/generic/string.h>

#include <map>
#include <memory>
#include <vector>

namespace NPassport::NLast {
    //
    // Abstract test case variable
    //
    class TVariator;
    class TVariable {
    public:
        TVariable() = default;
        virtual ~TVariable() = default;
        virtual bool empty() const = 0;
        virtual std::unique_ptr<TVariator> GetVariator() const = 0;
        virtual void Print(IOutputStream& s = Cout) const = 0;
    };

    //
    // Named test case variable instance -- base for a specific instance
    // of a named variable (such as CGI, Cookie and Header). Each isntance
    // associates a value with an instance ID.
    //
    class TNamedVarInst {
    public:
        TNamedVarInst(const TString& id)
            : Id_(id)
        {
        }
        virtual ~TNamedVarInst() = default;

        virtual const TString& Id() const {
            return Id_;
        }
        virtual TString GetValue() const = 0;

        virtual void Print(IOutputStream& s = Cout) const = 0;

    protected:
        TString Id_;
    };

    //
    // Static named test case variable instance - specific named test
    // case variables instances whose values are hardcoded in the scenario.
    //
    class TStaticNamedInst: public TNamedVarInst {
    public:
        TStaticNamedInst(const TString& id, const TString& value)
            : TNamedVarInst(id)
            , Value_(value)
        {
        }
        ~TStaticNamedInst() override = default;

        TString GetValue() const override {
            return Value_;
        }

        void Print(IOutputStream& s) const override;

    private:
        TString Value_;
    };

    //
    // Dynamic named test case variable instance -- specific named test case
    // variables whose values are determined at run-time by invoking a
    // value-generating function.
    //
    class TDynamicNamedInst: public TNamedVarInst {
    public:
        TDynamicNamedInst(const TString& id, const TString& fname)
            : TNamedVarInst(id)
            , Function_(fname)
        {
            NGen::TGenerateFunctions::Lookup(fname); // checks existence
        }
        ~TDynamicNamedInst() override = default;

        void AddArg(const TString& name, TArg&& arg) {
            Args_.insert(std::make_pair(name, std::move(arg)));
        }

        TString GetValue() const override;
        void Pregen(const TString& name);

        void Print(IOutputStream& s) const override;

    private:
        TString Function_;
        TFunctionArgs Args_;
        TString CgiName_;
        bool IsPregen_ = false;
    };

    //
    // Named test case variable (such as CGI or Cookie). Associates
    // variable name with a list of instances (ID/value pairs).
    //
    class TNamedVar: public TVariable {
    public:
        TNamedVar(const TString& name)
            : Name_(name)
        {
        }

        using TValues = std::vector<std::unique_ptr<TNamedVarInst>>;
        void AddValue(std::unique_ptr<TNamedVarInst> v) {
            Values_.push_back(std::move(v));
        }

        bool MatchInstance(const TString& name, const TNamedVarInst* value) const {
            if (name != Name_) {
                return false;
            }

            for (const auto& s : Values_) {
                if ((value == nullptr && s) || (value != nullptr && !s)) {
                    continue;
                }
                if (value == nullptr) {
                    return true;
                }
                if (value->Id() == s->Id()) {
                    return true;
                }
            }

            return false;
        }

        bool empty() const override {
            return Values_.empty();
        }
        void Print(IOutputStream& s) const override;

    protected:
        TString Name_;
        TValues Values_;
    };

    //
    // CGI variable
    //
    class TCgiVar: public TNamedVar {
    public:
        TCgiVar(const TString& name)
            : TNamedVar(name)
        {
        }
        std::unique_ptr<TVariator> GetVariator() const override;
        void Print(IOutputStream& s) const override;
    };

    class THeaderVar: public TNamedVar {
    public:
        THeaderVar(const TString& name)
            : TNamedVar(name)
        {
        }
        std::unique_ptr<TVariator> GetVariator() const override;
        void Print(IOutputStream& s) const override;
    };

    //
    // Cookie variable
    //
    class TCookieVar: public TNamedVar {
    public:
        TCookieVar(const TString& name)
            : TNamedVar(name)
        {
        }
        std::unique_ptr<TVariator> GetVariator() const override;
        void Print(IOutputStream& s) const override;
    };

    //
    // Path variable
    //
    class TPathVar: public TVariable {
    public:
        TPathVar() = default;

        using TValues = std::vector<std::unique_ptr<TString>>;
        void AddValue(std::unique_ptr<TString> v) {
            Values_.push_back(std::move(v));
        }
        void Print(IOutputStream& s) const override;
        bool empty() const override {
            return Values_.empty();
        }

        std::unique_ptr<TVariator> GetVariator() const override;

        bool MatchInstance(const TString* value) const {
            for (const auto& s : Values_) {
                if ((value == nullptr && s) || (value != nullptr && !s)) {
                    continue;
                }
                if (value == nullptr) {
                    return true;
                }
                if (*value == *s) {
                    return true;
                }
            }

            return false;
        }

    private:
        TValues Values_;
    };

    //
    // Abstract variator -- iterates over instances of a
    // test case variable
    //
    class TVarSet;
    class TVariator {
    public:
        TVariator() = default;
        virtual ~TVariator() = default;
        virtual bool Next() = 0;
        virtual void Apply(TTestContext&) = 0;
        virtual void AddSelf(TTestContext&) = 0;

        virtual bool MatchPath(const TPathVar*) const {
            return false;
        }
        virtual bool MatchCgi(const TNamedVar&) const {
            return false;
        }
        virtual bool MatchHeader(const TNamedVar&) const {
            return false;
        }
        virtual bool MatchCookie(const TNamedVar&) const {
            return false;
        }

        void MarkMatched() {
            AlreadyMatched_ = true;
        }
        void MarkUnmatched() {
            AlreadyMatched_ = false;
        }
        bool MarkedMatched() const {
            return AlreadyMatched_;
        }

        virtual void Print(IOutputStream& s = Cout) const = 0;

        virtual size_t Count() const = 0;

    protected:
        bool AlreadyMatched_ = false;
    };

    //
    // VarSet -- iterates over all possible test case variables combinations
    //
    class TCase;
    class TCheck;
    class TVarSet {
    public:
        TVarSet(const TString& url, const TCase&);
        ~TVarSet();

        bool Next(TTestContext&);
        size_t Count() const;

        using TValues = std::multimap<TString, TString>;

        void Print(IOutputStream& s = Cout) const;

    private:
        bool Match(const TCheck&);
        const TCase& Case_;

        using TVariators = std::vector<std::unique_ptr<TVariator>>;
        TVariators Variators_;

        TString Schema_;

        bool Complete_ = false;
    };

    class TSigner {
    public:
        static void Sign(TTestContext& ctx);

    private:
        static const TString& GetArg(TTestContext& ctx, const TString& arg, bool mandatory);
        static void SignTsRequest(TTestContext& ctx);
        static void SignTvm2Request(TTestContext& ctx);
    };

    //
    // Variator for a named variable
    //
    class TNamedVariator: public TVariator {
    public:
        TNamedVariator(const TString& name, const TNamedVar::TValues& values)
            : Name_(name)
            , Values_(values)
        {
            Current_ = Values_.begin();
        }

        bool Next() override {
            if (Current_ == Values_.end() || ++Current_ == Values_.end()) {
                Current_ = Values_.begin();
                return false;
            }
            return true;
        }

        void Print(IOutputStream& s) const override;

        size_t Count() const override {
            return Values_.size();
        }

    protected:
        TString Name_;
        const TNamedVar::TValues& Values_;
        TNamedVar::TValues::const_iterator Current_;
    };

    //
    // CGI variator
    //
    class TCgiVariator: public TNamedVariator {
    public:
        TCgiVariator(const TString& name, const TNamedVar::TValues& values)
            : TNamedVariator(name, values)
        {
        }

        void Apply(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Cgis.emplace(Name_, (*Current_)->GetValue());
            }
        }

        void AddSelf(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Vars.push_back(std::make_shared<TCgiVariator>(*this));
            }
        }

        bool MatchCgi(const TNamedVar& v) const override {
            if (AlreadyMatched_ || Current_ == Values_.end()) {
                return false;
            }
            return v.MatchInstance(Name_, Current_->get());
        }

        void Print(IOutputStream& s) const override;
    };

    //
    // Header variator
    //
    class THeaderVariator: public TNamedVariator {
    public:
        THeaderVariator(const TString& name, const TNamedVar::TValues& values)
            : TNamedVariator(name, values)
        {
        }

        void Apply(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                TString v((*Current_)->GetValue());
                ctx.Headers.insert(std::make_pair(Name_, v));
            }
        }

        void AddSelf(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Vars.push_back(std::make_shared<THeaderVariator>(*this));
            }
        }

        bool MatchHeader(const TNamedVar& v) const override {
            if (AlreadyMatched_ || Current_ == Values_.end()) {
                return false;
            }
            return v.MatchInstance(Name_, Current_->get());
        }

        void Print(IOutputStream& s) const override;
    };

    //
    // Cookie variator
    //
    class TCookieVariator: public TNamedVariator {
    public:
        TCookieVariator(const TString& name, const TNamedVar::TValues& values)
            : TNamedVariator(name, values)
        {
        }

        void Apply(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                TString v((*Current_)->GetValue());
                ctx.Cookies.insert(std::make_pair(Name_, v));
            }
        }

        void AddSelf(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Vars.push_back(std::make_shared<TCookieVariator>(*this));
            }
        }

        bool MatchCookie(const TNamedVar& v) const override {
            if (AlreadyMatched_ || Current_ == Values_.end()) {
                return false;
            }
            return v.MatchInstance(Name_, Current_->get());
        }

        void Print(IOutputStream& s) const override;
    };

    //
    // Path variator
    //
    class TPathVariator: public TVariator {
    public:
        TPathVariator(const TPathVar::TValues& values)
            : Values_(values)
        {
            Current_ = Values_.begin();
        }

        bool Next() override {
            if (Current_ == Values_.end() || ++Current_ == Values_.end()) {
                Current_ = Values_.begin();
                return false;
            }
            return true;
        }

        void Apply(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Path = *(*Current_);
            }
        }

        void AddSelf(TTestContext& ctx) override {
            assert(Current_ != Values_.end());
            if (*Current_ != nullptr) {
                ctx.Vars.push_back(std::make_shared<TPathVariator>(*this));
            }
        }

        bool MatchPath(const TPathVar* v) const override {
            if (AlreadyMatched_ || Current_ == Values_.end()) {
                return false;
            }
            return v->MatchInstance(Current_->get());
        }

        void Print(IOutputStream& s) const override;

        size_t Count() const override {
            return Values_.size();
        }

    private:
        const TPathVar::TValues& Values_;
        TPathVar::TValues::const_iterator Current_;
    };

}
