package blackbox

import (
	"context"
	"fmt"
	"strconv"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb/bbtypes"
)

type FindPddAccountsBy string
type FindPddAccountsOrder string

type FindPddAccountsSort struct {
	Key   FindPddAccountsBy
	Order FindPddAccountsOrder
}

const (
	FindPddAccountsKeyUID                         = "uid"
	FindPddAccountsKeyLogin                       = "login"
	FindPddAccountsOrderAsc  FindPddAccountsOrder = "asc"
	FindPddAccountsOrderDesc FindPddAccountsOrder = "desc"
)

type FindPddAccountsRequest struct {
	DomainID uint64
	Offset   uint64
	Limit    uint64
}
type RawFindPddAccountsResponse struct {
	bbtypes.BaseResponse
	UIDs       []string `json:"uids"`
	Count      string   `json:"count"`
	TotalCount string   `json:"total_count"`
}
type FindPddAccountsResponse struct {
	bbtypes.BaseResponse
	UIDs       []blackbox.ID
	Count      uint64
	TotalCount uint64
}

func (c *blackboxClient) FindPddAccounts(ctx context.Context, domainID uint64, offset uint64, limit uint64, sortParams FindPddAccountsSort) (*FindPddAccountsResponse, error) {
	var (
		sortKey   string
		sortOrder string
	)
	switch sortParams.Key {
	case FindPddAccountsKeyUID:
		sortKey = "uid"
	case FindPddAccountsKeyLogin:
		sortKey = "login"
	default:
		return nil, fmt.Errorf("unexpected sort key: %s", sortParams.Key)
	}
	switch sortParams.Order {
	case FindPddAccountsOrderAsc:
		sortOrder = "asc"
	case FindPddAccountsOrderDesc:
		sortOrder = "desc"
	default:
		return nil, fmt.Errorf("unexpected sort sortParams: %s", sortParams.Order)
	}

	httpReq := c.InternalHTTP().R(ctx).SetQueryParams(map[string]string{
		"method":    "find_pdd_accounts",
		"format":    "json",
		"domain_id": fmt.Sprintf("%d", domainID),
		"offset":    fmt.Sprintf("%d", offset),
		"limit":     fmt.Sprintf("%d", limit),
		"sort":      sortKey + "." + sortOrder,
	})

	var rsp RawFindPddAccountsResponse
	if err := c.InternalHTTP().Get(httpReq, &rsp); err != nil {
		return nil, err
	}
	var retval FindPddAccountsResponse
	retval.BaseResponse.Error = rsp.BaseResponse.Error
	retval.BaseResponse.Exception = rsp.BaseResponse.Exception
	if count, err := strconv.ParseUint(rsp.Count, 10, 64); err != nil {
		return nil, err
	} else {
		retval.Count = count
	}
	if totalCount, err := strconv.ParseUint(rsp.TotalCount, 10, 64); err != nil {
		return nil, err
	} else {
		retval.TotalCount = totalCount
	}

	for _, rawUID := range rsp.UIDs {
		if uid, err := strconv.ParseUint(rawUID, 10, 64); err != nil {
			return nil, err
		} else {
			retval.UIDs = append(retval.UIDs, uid)
		}
	}
	return &retval, nil
}
