package blackbox

import (
	"context"
	"fmt"
	"strconv"

	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb/bbtypes"
)

type HostedDomainsRequest struct {
	DomainID uint64
}

type RawHostedDomain struct {
	AdminUID     string `json:"admin"`
	BornDate     string `json:"born_date"`
	DefaultUID   string `json:"default_uid"`
	Domain       string `json:"domain"`
	DomainID     string `json:"domid"`
	Enabled      string `json:"ena"`
	MasterDomain string `json:"master_domain"`
	MX           string `json:"mx"`
	Options      string `json:"options"`
}
type RawHostedDomainsResponse struct {
	bbtypes.BaseResponse
	Domains []RawHostedDomain `json:"hosted_domains"`
}
type HostedDomain struct {
	AdminUID     uint64
	BornDate     string
	DefaultUID   uint64
	Domain       string
	DomainID     uint64
	Enabled      bool
	MasterDomain string
	MX           bool
	Options      string
}
type HostedDomainsResponse struct {
	bbtypes.BaseResponse
	Domains []HostedDomain
}

func (c *blackboxClient) HostedDomains(ctx context.Context, domainID uint64) (*HostedDomainsResponse, error) {
	httpReq := c.InternalHTTP().R(ctx).SetQueryParams(map[string]string{
		"method":    "hosted_domains",
		"format":    "json",
		"domain_id": fmt.Sprintf("%d", domainID),
	})

	var rsp RawHostedDomainsResponse
	if err := c.InternalHTTP().Get(httpReq, &rsp); err != nil {
		return nil, err
	}
	var retval HostedDomainsResponse
	retval.BaseResponse.Error = rsp.BaseResponse.Error
	retval.BaseResponse.Exception = rsp.BaseResponse.Exception
	for _, rawDomain := range rsp.Domains {
		var (
			domain HostedDomain
		)
		if len(rawDomain.AdminUID) != 0 {
			if adminUID, err := strconv.ParseUint(rawDomain.AdminUID, 10, 64); err != nil {
				return nil, err
			} else {
				domain.AdminUID = adminUID
			}
		}
		if len(rawDomain.DefaultUID) != 0 {
			if defaultUID, err := strconv.ParseUint(rawDomain.DefaultUID, 10, 64); err != nil {
				return nil, err
			} else {
				domain.DefaultUID = defaultUID
			}
		}
		if len(rawDomain.DomainID) != 0 {
			if domainID, err := strconv.ParseUint(rawDomain.DomainID, 10, 64); err != nil {
				return nil, err
			} else {
				domain.DomainID = domainID
			}
		}
		if len(rawDomain.Enabled) != 0 {
			enabled, err := strconv.ParseBool(rawDomain.Enabled)
			if err != nil {
				return nil, err
			}
			domain.Enabled = enabled
		}
		if len(rawDomain.MX) != 0 {
			mx, err := strconv.ParseBool(rawDomain.MX)
			if err != nil {
				return nil, err
			}
			domain.MX = mx
		}
		domain.BornDate = rawDomain.BornDate
		domain.Domain = rawDomain.Domain
		domain.MasterDomain = rawDomain.MasterDomain
		domain.Options = rawDomain.Options
		retval.Domains = append(retval.Domains, domain)
	}

	return &retval, nil
}
