package httpdaemon

import (
	"fmt"
	"net/http"
	"os"

	"github.com/labstack/echo/v4"
	"github.com/shirou/gopsutil/v3/process"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type httpUnistatConfig struct {
	Address string `json:"listen_address"`
	Port    uint16 `json:"port"`
}

func startUnistatHTTP(cfg *httpUnistatConfig, stopper chan<- bool, echo *echo.Echo) (*httpState, error) {
	state := &httpState{
		echo: echo,
		addr: fmt.Sprintf("%s:%d", cfg.Address, cfg.Port),
	}

	go func() {
		logger.Log().Infof("Http server (unistat) started: '%s'", state.addr)
		err := state.echo.Start(state.addr)
		if err != nil && err != http.ErrServerClosed {
			logger.Log().Errorf("Http server (unistat) stopped: %s", err)
			stopper <- true
		}
	}()

	return state, nil
}

func handleUnistat(d *DefaultUnistat) echo.HandlerFunc {
	return func(c echo.Context) error {
		d.Update()
		return c.JSONBlob(http.StatusOK, unistat.DefaultChunk.Serialize())
	}
}

type DefaultUnistat struct {
	allRequests    *unistat.SignalDiff
	onlineRequests *unistat.SignalAbsolute
	failRequests   *unistat.SignalDiff
	requestsByPath *unistat.SignalSet

	virtualMemory  *unistat.SignalAbsolute
	residentMemory *unistat.SignalAbsolute
	threads        *unistat.SignalAbsolute

	selfProc *process.Process
}

func NewDefaultUnistat() (*DefaultUnistat, error) {
	selfProc, err := process.NewProcess(int32(os.Getpid()))
	if err != nil {
		return nil, xerrors.Errorf("Failed to get process info: %s", err)
	}

	// Signal names are the same as in C++ daemons
	return &DefaultUnistat{
		allRequests: unistat.DefaultChunk.CreateSignalDiff("http.requests_1"),
		onlineRequests: unistat.DefaultChunk.CreateSignalAbsoluteAverage(
			"http.threads.busy_1"),
		failRequests:   unistat.DefaultChunk.CreateSignalDiff("http.threads.fail_1"),
		requestsByPath: unistat.DefaultChunk.CreateSignalSet("in.requests."),
		virtualMemory: unistat.DefaultChunk.CreateSignalAbsoluteAverage(
			"proc.self.stat.virtual_memory"),
		residentMemory: unistat.DefaultChunk.CreateSignalAbsoluteAverage(
			"proc.self.stat.resident_memory"),
		threads: unistat.DefaultChunk.CreateSignalAbsoluteAverage(
			"proc.self.stat.num_threads"),
		selfProc: selfProc,
	}, nil
}

func (d *DefaultUnistat) Update() {
	memoryInfo, err := d.selfProc.MemoryInfo()
	if err == nil {
		d.virtualMemory.Store(float64(memoryInfo.VMS))
		d.residentMemory.Store(float64(memoryInfo.RSS))
	} else {
		d.virtualMemory.Store(0)
		d.residentMemory.Store(0)
	}

	numThreads, err := d.selfProc.NumThreads()
	if err == nil {
		d.threads.Store(float64(numThreads))
	} else {
		d.threads.Store(0)
	}
}
