package httpdtvm

import (
	"io/ioutil"
	"os"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type TvmConfig struct {
	SelfID                 tvm.ClientID            `json:"self_tvm_id"`
	CacheDir               string                  `json:"disk_cache_dir"`
	CheckServiceTickets    bool                    `json:"check_service_tickets"`
	BlackboxEnv            string                  `json:"blackbox_env"`
	RolesForIdmSlug        string                  `json:"roles_for_idm_slug"`
	DisableSrcCheck        bool                    `json:"disable_src_check"`
	DisableDefaultUIDCheck bool                    `json:"disable_default_uid_check"`
	SecretEnvVar           string                  `json:"secret_env_var"`
	SecretFilePath         string                  `json:"secret_filepath"`
	Destinations           map[string]tvm.ClientID `json:"destinations"`

	// only for tests

	TVMHost     string       `json:"tvm_host"`
	TVMPort     int          `json:"tvm_port"`
	TiroleHost  string       `json:"tirole_host"`
	TirolePort  int          `json:"tirole_port"`
	TiroleTvmID tvm.ClientID `json:"tirole_tvm_id"`
}

func InitTvm(cfg TvmConfig) (*tvmauth.Client, error) {
	settings, err := readSettings(cfg)
	if err != nil {
		return nil, err
	}

	return tvmauth.NewAPIClient(
		*settings,
		logger.Log().With(log.Nil("TvmClient")),
	)
}

func readSettings(cfg TvmConfig) (*tvmauth.TvmAPISettings, error) {
	env, err := convertBbEnv(cfg.BlackboxEnv)
	if err != nil {
		return nil, err
	}

	secret, err := readSecret(cfg.SecretEnvVar, cfg.SecretFilePath)
	if err != nil {
		return nil, err
	}

	if cfg.CacheDir == "" {
		return nil, xerrors.Errorf("tvm cache dir is not configured")
	}

	settings := &tvmauth.TvmAPISettings{
		SelfID:                      cfg.SelfID,
		EnableServiceTicketChecking: cfg.CheckServiceTickets,
		BlackboxEnv:                 env,
		FetchRolesForIdmSystemSlug:  cfg.RolesForIdmSlug,
		DisableSrcCheck:             cfg.DisableSrcCheck,
		DisableDefaultUIDCheck:      cfg.DisableDefaultUIDCheck,
		DiskCacheDir:                cfg.CacheDir,
		TVMHost:                     cfg.TVMHost,
		TVMPort:                     cfg.TVMPort,
		TiroleHost:                  cfg.TiroleHost,
		TirolePort:                  cfg.TirolePort,
		TiroleTvmID:                 cfg.TiroleTvmID,
	}
	if secret != "" || len(cfg.Destinations) != 0 {
		settings.ServiceTicketOptions = tvmauth.NewAliasesOptions(
			secret,
			cfg.Destinations,
		)
	}

	return settings, nil
}

func convertBbEnv(envStr string) (*tvm.BlackboxEnv, error) {
	if envStr == "" {
		return nil, nil
	}

	res, err := tvm.BlackboxEnvFromString(envStr)
	return &res, err
}

func readSecret(env, filepath string) (string, error) {
	if env == "" && filepath == "" {
		return "", nil
	}

	if env != "" && filepath != "" {
		return "", xerrors.Errorf(
			"options 'secret_env_var' and 'secret_filepath' conflict with each other: you should provide only one option",
		)
	}

	if env != "" {
		secret, _ := os.LookupEnv(env)
		if secret == "" {
			return "", xerrors.Errorf("Failed to get TVM client secret from env var '%s'", env)
		}
		return secret, nil
	}

	secret, err := ioutil.ReadFile(filepath)
	if err != nil {
		return "", xerrors.Errorf("Failed to read TVM secret: %w", err)
	}

	return string(secret), nil
}
