package httpdaemon

import (
	"fmt"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

func (d *DefaultUnistat) MiddlewareRequests() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			d.allRequests.Inc()

			bh := unistat.CreateBusyHolder(d.onlineRequests)

			err = next(c)
			if err != nil {
				d.failRequests.Inc()
			}

			bh.Release()

			return err
		}
	}
}

func createHandleKey(method, path string) string {
	return fmt.Sprintf("%s.%s", method, path)
}

func (d *DefaultUnistat) MiddlewareRequestsByPath(routes []*echo.Route) echo.MiddlewareFunc {
	for _, r := range routes {
		d.requestsByPath.CreateSignal(createHandleKey(r.Method, r.Path))
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			d.requestsByPath.IncSignal(createHandleKey(c.Request().Method, c.Request().URL.Path))
			return next(c)
		}
	}
}

func MiddlewareHTTPCodes() echo.MiddlewareFunc {
	codes := unistat.DefaultChunk.CreateSignalSet("code.")
	codes.CreateSignal("200")
	codes.CreateSignal("400")
	codes.CreateSignal("500")
	codes.CreateSignal("502")
	codes.CreateSignal("503")
	codes.CreateSignal("504")

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			err = next(c)
			codes.CreateOrIncSignal(fmt.Sprintf("%d", c.Response().Status))
			return err
		}
	}
}

func MiddlewareResponseTime() echo.MiddlewareFunc {
	msec := []int{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 15, 17, 20, 25, 30, 35, 40, 45, 50, 60, 70, 80, 90, 100, 125, 150, 175, 200, 225, 250, 275, 300, 400, 500, 750, 1000, 2000, 3000}
	bounds := make([]time.Duration, len(msec))
	for idx := range msec {
		bounds[idx] = time.Duration(msec[idx]) * time.Millisecond
	}

	times, err := unistat.DefaultChunk.CreateTimeStats("full_answer_time", bounds)
	if err != nil {
		panic(err)
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			start := time.Now()
			err = next(c)

			times.Insert(time.Since(start))

			return err
		}
	}
}
