package httpdaemon

import (
	"encoding/json"
	"io/ioutil"
	"math/rand"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	TZMoscow = "Europe/Moscow"
)

func Run(cfgFile string, factory ServiceFactory) error {
	if err := os.Setenv("TZ", TZMoscow); err != nil {
		return xerrors.Errorf("Failed to set TZ to '%s': %w", TZMoscow, err)
	}

	// I.e. to provide "unique" request_id after restart
	rand.Seed(time.Now().Unix())

	cfg, err := parseConfig(cfgFile)
	if err != nil {
		return err
	}

	if err := logger.Init(cfg.Log); err != nil {
		return err
	}

	logger.Log().Info("Starting server")
	serv, err := factory.NewService(cfg.ServiceCfg)
	if err != nil {
		logger.Log().Infof("Failed to create service: %s", err)
		return err
	}
	opts := &Options{}
	if s, ok := serv.(ServiceWithOptions); ok {
		opts = s.GetOptions()
	}

	defaultUnistat, err := NewDefaultUnistat()
	if err != nil {
		logger.Log().Infof("Failed to create default unistat registry: %s", err)
		return err
	}

	echoCommon := echo.New()
	echoCommon.HideBanner = true
	// Use header 'X-Real-IP' which got only from loopback
	echoCommon.IPExtractor = echo.ExtractIPFromRealIPHeader(echo.TrustLoopback(true))
	echoCommon.Pre(
		middlewares.StartInstant(),
		MiddlewareResponseTime(),
		middlewares.ReqID(),
		defaultUnistat.MiddlewareRequests(),
		MiddlewareHTTPCodes(),
	)
	serv.AddHandlers(echoCommon)
	if !opts.DisableSignalsByPath {
		echoCommon.Use(defaultUnistat.MiddlewareRequestsByPath(echoCommon.Routes()))
	}

	echoUnistat := echo.New()
	echoUnistat.HideBanner = true
	echoUnistat.GET("/unistat", handleUnistat(defaultUnistat))
	if opts.ExtendedUnistatHandler != nil {
		opts.ExtendedUnistatHandler(echoUnistat)
	}

	stopChannel := make(chan bool)

	if cfg.HTTPConfig == nil && cfg.HTTPSConfig == nil {
		return xerrors.Errorf("'http_common' and 'https_common' in config are empty")
	}

	logger.Log().Info("Starting http")
	servers, err := runHTTPInterfaces(cfg, stopChannel, echoCommon, echoUnistat)
	if err != nil {
		logger.Log().Infof("Failed to start http: %s", err)
		return err
	}

	logger.Log().Info("Started")

	signalChannel := make(chan os.Signal, 10)
	signal.Notify(signalChannel, syscall.SIGQUIT, syscall.SIGINT, syscall.SIGTERM)

	select {
	case sig := <-signalChannel:
		switch sig {
		case syscall.SIGTERM:
			logger.Log().Info("Stopping: SIGTERM")
		case syscall.SIGINT:
			logger.Log().Info("Stopping: SIGINT")
		case syscall.SIGQUIT:
			logger.Log().Info("Stopping: SIGQUIT")
		}

	case <-stopChannel:
		logger.Log().Info("Stopping: internal reason")
	}

	for _, s := range servers {
		s.stop()
	}
	if opts.StopService != nil {
		opts.StopService()
	}

	return nil
}

func parseConfig(cfgFile string) (*Config, error) {
	data, err := ioutil.ReadFile(cfgFile)
	if err != nil {
		return nil, err
	}

	var cfg Config
	if err := json.Unmarshal(data, &cfg); err != nil {
		return nil, xerrors.Errorf("failed to parse Config: %w", err)
	}

	return &cfg, nil
}

func runHTTPInterfaces(cfg *Config, stopChannel chan<- bool, echoCommon *echo.Echo, echoUnistat *echo.Echo) ([]*httpState, error) {
	servers := make([]*httpState, 0, 3)

	if cfg.HTTPConfig == nil {
		logger.Log().Info("Skipping http common interface: it is not configured")
	} else {
		httpCommon, err := startCommonHTTP(cfg.HTTPConfig, stopChannel, echoCommon)
		if err != nil {
			return nil, xerrors.Errorf("Failed to start http common: %v", err)
		}
		servers = append(servers, httpCommon)
	}

	if cfg.HTTPSConfig == nil {
		logger.Log().Info("Skipping https common interface: it is not configured")
	} else {
		httpCommon, err := startCommonHTTPS(cfg.HTTPSConfig, stopChannel, echoCommon)
		if err != nil {
			return nil, xerrors.Errorf("Failed to start https common: %v", err)
		}
		servers = append(servers, httpCommon)
	}

	if cfg.HTTPUnistat == nil {
		logger.Log().Info("Skipping http unistat interface: it is not configured")
	} else {
		httpUnistat, err := startUnistatHTTP(cfg.HTTPUnistat, stopChannel, echoUnistat)
		if err != nil {
			return nil, xerrors.Errorf("Failed to start http unistat: %v", err)
		}
		servers = append(servers, httpUnistat)
	}

	return servers, nil
}
