package juggler

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/yandex/tvm"
)

type Code uint16

const (
	Ok       Code = 0
	Warning  Code = 1
	Critical Code = 2
)

type Status struct {
	Code    Code
	Message string
}

func NewStatusOk() *Status {
	return NewStatus(Ok, "OK")
}

func NewStatus(code Code, format string, args ...interface{}) *Status {
	return &Status{
		Code:    code,
		Message: fmt.Sprintf(format, args...),
	}
}

func TvmClientStatus(client tvm.Client) *Status {
	tvmClientStatus, err := client.GetStatus(context.Background())
	if err != nil {
		return NewStatus(Critical, "Failed to check tvm client status")
	}

	var code Code
	switch tvmClientStatus.Status {
	case tvm.ClientOK:
		code = Ok
	case tvm.ClientWarning:
		code = Warning
	case tvm.ClientError:
		code = Critical
	default:
		code = Critical
	}
	return NewStatus(code, tvmClientStatus.LastError)
}

func (status *Status) Update(other *Status) {
	status.Updatef(other.Code, other.Message)
}

func (status *Status) Updatef(code Code, format string, args ...interface{}) {
	if code < status.Code {
		return
	}

	status.Code = code
	status.Message = fmt.Sprintf(format, args...)
}

func (status *Status) String() string {
	return fmt.Sprintf("%d;%s", status.Code, status.Message)
}
