package logger

import (
	"fmt"
	"syscall"
	"time"

	origZap "go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"
)

type LogFormat string

const (
	LogPlaintext LogFormat = "plaintext"
	LogDeploy    LogFormat = "deploy"
)

type Config struct {
	FilePath             string    `json:"file"`
	DisablePrintingLevel bool      `json:"disable_printing_level"`
	DisablePrintingTime  bool      `json:"disable_printing_time"`
	Format               LogFormat `json:"log_format"`
	LogLevel             string    `json:"level"`
}

var l *zap.Logger

func init() {
	_ = logrotate.RegisterLogrotateSink(syscall.SIGHUP)
	_ = Init(Config{FilePath: "_NOLOG_"})
}

func Init(cfg Config) error {
	logger, err := CreateLog(cfg)
	if err != nil {
		return fmt.Errorf("failed to initialize logger: %s", err)
	}

	l = logger
	return nil
}

func getLoggerLevel(cfg Config) (log.Level, error) {
	var textLevel string
	if cfg.LogLevel != "" {
		textLevel = cfg.LogLevel
	} else {
		textLevel = log.DebugString
	}
	ll, err := log.ParseLevel(textLevel)
	return ll, err
}

func CreateLog(cfg Config) (*zap.Logger, error) {
	if cfg.Format == LogPlaintext || cfg.Format == "" {
		return createConsoleLogger(cfg)
	} else if cfg.Format == LogDeploy {
		return createDeployLogger(cfg)
	} else {
		return nil, fmt.Errorf("unexpected log format value: %s", cfg.Format)
	}
}

func createConsoleLogger(cfg Config) (*zap.Logger, error) {
	level, err := getLoggerLevel(cfg)
	if err != nil {
		return nil, err
	}
	loggerCfg := zap.ConsoleConfig(level)
	loggerCfg.DisableCaller = true
	loggerCfg.DisableStacktrace = true

	if cfg.DisablePrintingLevel {
		loggerCfg.EncoderConfig.EncodeLevel = func(zapcore.Level, zapcore.PrimitiveArrayEncoder) {}
	}

	if cfg.DisablePrintingTime {
		loggerCfg.EncoderConfig.EncodeTime = func(time.Time, zapcore.PrimitiveArrayEncoder) {}
	}

	if cfg.FilePath != "_NOLOG_" {
		loggerCfg.OutputPaths = []string{"logrotate://" + cfg.FilePath}
	}

	res, err := zap.New(loggerCfg)
	if err != nil {
		return nil, fmt.Errorf("failed to initialize logger: %s", err)
	}

	return res, err
}

func createDeployLogger(cfg Config) (*zap.Logger, error) {
	level, err := getLoggerLevel(cfg)
	if err != nil {
		return nil, err
	}
	loggerCfg := zap.NewDeployConfig()
	loggerCfg.Level = origZap.NewAtomicLevelAt(zap.ZapifyLevel(level))
	loggerCfg.DisableCaller = true
	loggerCfg.DisableStacktrace = true

	if cfg.FilePath != "_NOLOG_" {
		loggerCfg.OutputPaths = []string{"logrotate://" + cfg.FilePath}
	}
	res, err := zap.New(loggerCfg)
	if err != nil {
		return nil, fmt.Errorf("failed to initialize logger: %s", err)
	}

	return res, nil
}

func Log() *zap.Logger {
	return l
}
