package unistat

import (
	"time"

	"go.uber.org/atomic"

	libunistat "a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

type Chunk struct {
	registry libunistat.Registry
}

var DefaultChunk = NewChunk()

func NewChunk() *Chunk {
	return &Chunk{
		registry: libunistat.NewRegistry(),
	}
}

func (c *Chunk) CreateSignalDiff(name string) *SignalDiff {
	res := &SignalDiff{
		metric: libunistat.NewNumeric(name, 1, aggr.Counter(), libunistat.Sum),
	}

	c.registry.Register(res.metric)

	return res
}

func (c *Chunk) CreateSignalAbsoluteMax(name string) *SignalAbsolute {
	res := &SignalAbsolute{
		metric: libunistat.NewNumeric(
			name,
			1,
			&libunistat.StructuredAggregation{
				AggregationType: libunistat.Absolute,
				Group:           libunistat.Max,
				MetaGroup:       libunistat.Max,
				Rollup:          libunistat.Max,
			},
			libunistat.Last,
		),
		value: atomic.NewFloat64(0),
	}

	c.registry.Register(res.metric)

	return res
}

func (c *Chunk) CreateSignalAbsoluteAverage(name string) *SignalAbsolute {
	res := &SignalAbsolute{
		metric: libunistat.NewNumeric(
			name,
			1,
			&libunistat.StructuredAggregation{
				AggregationType: libunistat.Absolute,
				Group:           libunistat.Average,
				MetaGroup:       libunistat.Average,
				Rollup:          libunistat.Average,
			},
			libunistat.Last,
		),
		value: atomic.NewFloat64(0),
	}

	c.registry.Register(res.metric)

	return res
}

func (c *Chunk) CreateSignalSet(prefix string) *SignalSet {
	return newSignalSet(prefix, c)
}

func (c *Chunk) CreateSignalHgram(name string, bounds []float64) (*SignalHgram, error) {
	res, err := newSignalHgram(name, bounds)
	if err != nil {
		return nil, err
	}

	c.registry.Register(res.metric)

	return res, nil
}

func (c *Chunk) CreateTimeStats(name string, bounds []time.Duration) (*TimeStat, error) {
	nums := make([]float64, 0)
	for _, d := range bounds {
		nums = append(nums, float64(d.Milliseconds()))
	}

	hgram, err := c.CreateSignalHgram(name, nums)
	if err != nil {
		return nil, err
	}

	return newTimeStat(hgram), nil
}

func (c *Chunk) Serialize() []byte {
	res, err := c.registry.MarshalJSON()
	if err != nil {
		panic(err)
	}

	return res
}
