package unistatresty

import (
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type Stats struct {
	ResponseTimings *unistat.TimeStat
	Requests        *unistat.SignalDiff
	NetworkErrs     *unistat.SignalDiff
}

type Config struct {
	Client        *resty.Client
	UnistatPrefix string

	// optional

	MaxResponseTime time.Duration
	Chunk           *unistat.Chunk
}

func PrepareStats(cfg Config) *Stats {
	if cfg.MaxResponseTime.Milliseconds() == 0 {
		cfg.MaxResponseTime = 3 * time.Second
	}
	if cfg.Chunk == nil {
		cfg.Chunk = unistat.DefaultChunk
	}

	responseTimings, err := cfg.Chunk.CreateTimeStats(
		cfg.UnistatPrefix+".response_time",
		unistat.CreateTimeBoundsFromMaxValue(cfg.MaxResponseTime),
	)
	if err != nil {
		panic(err)
	}

	stats := &Stats{
		ResponseTimings: responseTimings,
		Requests:        cfg.Chunk.CreateSignalDiff(cfg.UnistatPrefix + ".requests"),
		NetworkErrs:     cfg.Chunk.CreateSignalDiff(cfg.UnistatPrefix + ".network_errors"),
	}

	cfg.Client.
		OnBeforeRequest(func(c *resty.Client, r *resty.Request) error {
			stats.Requests.Inc()
			r.SetContext(middlewares.WithStartInstant(r.Context(), time.Now()))
			return nil
		}).
		OnAfterResponse(func(c *resty.Client, r *resty.Response) error {
			stats.ResponseTimings.Insert(
				time.Since(middlewares.ContextStartInstant(r.Request.Context())))
			return nil
		}).
		OnError(func(*resty.Request, error) {
			stats.NetworkErrs.Inc()
		})

	return stats
}
