#!/usr/bin/env python3
import argparse
import os

from datetime import datetime
import requests
from jinja2 import FileSystemLoader, Environment
import json

from startrek_client import Startrek

QUEUE = "PAYSYSADMIN"
DC_LETTERS = {"IVA": "e", "MYT": "f", "SAS": "h", "VLA": "v", "MAN": "w"}
# TODO: create ticket on next tuesday
# TODO: if ticket already exists, skip creation (idempotent behavior)
# TODO: assign ticket to responsible on next week for drills
# TODO: find duty devs in service and add them as followers and summon them for attempting
# TODO: send notification to infra date and time
# TODO: add link to epic ticket
# TODO: add link to previous closed drills ticket
# TODO: add ticket to next sprint if sprint doesn't exist create it
# TODO: add to calendar meeting with description


def main():
    # For testing
    api = "st-api.test.yandex-team.ru"
    web = "st.test.yandex-team.ru"
    infra_api = "https://infra-api-test.yandex-team.ru/v1"
    infra_service_id = 1885
    infra_env_id = 2969

    # for Prod
    # api = "st-api.yandex-team.ru"
    # web = "st.yandex-team.ru"
    # infra_api = "https://infra-api.yandex-team.ru/v1"
    # infra_service_id = 1885
    # infra_env_id = 2968

    parser = argparse.ArgumentParser(description="drills helper")
    parser.add_argument(
        "-dc", choices=["MAN", "SAS", "VLA", "MYT", "IVA"],
    )

    parser.add_argument("-start", nargs=2)
    parser.add_argument("-end", nargs=2)
    p = parser.parse_args()
    if not os.getenv("ST_TOKEN"):
        raise ValueError(
            "no ST_TOKEN, to get it go to https://oauth.yandex-team.ru/authorize?response_type=token&client_id=5f671d781aca402ab7460fde4050267b"
        )
    drills_ticket = DrillsTicket(p.dc, p.start, p.end, web, api)
    drills_ticket._create_ticket()
    # https: // infra - api - test.yandex - team.ru / swagger /  # /Events/post_events
    token = os.getenv("INFRA_TOKEN")
    if not os.getenv("INFRA_TOKEN"):
        raise ValueError(
            "no INFRA_TOKEN, to get it go to https://oauth.yandex-team.ru/authorize?response_type=token&client_id=5f671d781aca402ab7460fde4050267b"
        )
    print(drills_ticket.ticket)
    infra_cli = InfraClient(token, infra_api)
    print(infra_cli)

    event = InfraEvent(
        infra_env_id,
        infra_service_id,
        p.start,
        p.dc,
        "maintenance",
        "minor",
        "Внутренние учения",
        end=p.end,
    )
    infra_cli.send_event(event.event_data)


class DrillsTicket:
    def __init__(self, datacenter, start, end, web, api, infra_event=True):
        self.datacenter = datacenter
        self.start = " ".join(start)
        self.end = " ".join(end)
        self.web = web
        self.api = api
        self.ticket = None
        self.infra_event = infra_event

    def __validate_date(self):
        format = "%Y-%m-%d %H:%M"
        try:
            datetime.strptime(self.start, format)
            datetime.strptime(self.end, format)
        except ValueError:
            print("Wrong format")
            exit(1)

    @classmethod
    def __parse_templates(cls):
        file_loader = FileSystemLoader("templates")
        env = Environment(loader=file_loader)
        return env

    def _create_ticket(self):
        e = self.__parse_templates()
        self.__validate_date()
        template_summary = e.get_template("summary.tmpl")
        template_description = e.get_template("description.tmpl")
        summary = template_summary.render(
            datacenter=self.datacenter, date=[self.start, self.end]
        )
        description = template_description.render(
            datacenter=self.datacenter, dc_letter=DC_LETTERS[self.datacenter]
        )
        client = Startrek(
            useragent=os.path.basename(__file__),
            base_url="https://{}/v2".format(self.api),
            token=os.getenv("ST_TOKEN"),
        )
        issue = client.issues.create(
            queue=QUEUE,
            summary=summary,
            type={"name": "Task"},
            description=description,
            tags="DRILLS",
        )
        self.ticket = "https://{}/{}".format(self.web, issue.key)


class InfraClient:
    def __init__(self, token, api):
        self.token = token
        self.api = api

    def send_event(self, payload):
        headers = {
            "Content-type": "application/json",
            "Authorization": "OAuth {}".format(self.token),
        }
        print(headers)
        print(json.dumps(payload))
        r = requests.post(
            self.api + "/events", data=json.dumps(payload), headers=headers
        )
        # print(self.api)
        # TODO: return link
        print(r.text)
        return r.text


class InfraEvent:
    def __init__(
        self,
        environment_id,
        service_id,
        start,
        dc,
        issue_type,
        issue_severity,
        title,
        end=None,
        description=None,
        tickets=None,
        send_email=True,
    ):
        self.title = title
        self.description = description
        self.environment_id = environment_id
        self.service_id = service_id

        self.start = start
        self.end = end
        self.issue_type = issue_type
        self.issue_severity = issue_severity

        self.dc = dc
        self.tickets = tickets
        self.send_email = send_email
        self.event_data = self._prepare_event()

    def _prepare_event(self):
        event = dict()
        event["title"] = self.title
        event["description"] = self.description
        event["environmentId"] = self.environment_id
        event["serviceId"] = self.service_id
        event["startTime"] = datetime.strptime(
            " ".join(self.start), "%Y-%m-%d %H:%M"
        ).strftime("%s")
        event["finishTime"] = self.end
        event["severity"] = self.issue_severity
        event["type"] = self.issue_type
        for dc in ["iva", "sas", "myt", "man", "vla"]:
            if dc == self.dc.lower():
                event[dc] = True
            else:
                event[dc] = False
        event["tickets"] = self.tickets
        event["meta"] = {}
        event["sendEmailNotifications"] = self.send_email
        event["setAllAvailableDc"] = False

        print(json.dumps(event))
        if self.end:
            event["finishTime"] = datetime.strptime(
                " ".join(self.end), "%Y-%m-%d %H:%M"
            ).strftime("%s")
            print(self.start)
            print("EVENT: {}".format(event))
        return event

    def update_event(self):
        raise NotImplementedError

    def delete_event(self):
        raise NotImplementedError

    def get_current_events(self):
        raise NotImplementedError


if __name__ == "__main__":
    main()
