import json
from flask import jsonify, request, Blueprint
import mongo_idm.helpers
import mongo_idm.db
from mongo_idm.config import DEFAULT_DB_ROLES, DEFAULT_ROLES_INFO


api = Blueprint("api", __name__)


def info():
    info = DEFAULT_ROLES_INFO

    for db_id in mongo_idm.helpers.get_databases():

        info["values"][db_id] = {
            "name": {
                "ru": db_id,
                "en": db_id,
            },
            "roles": {
                "values": DEFAULT_DB_ROLES,
                "slug": "role",
            },
        }

    return info


@api.route("/info/", methods=["GET"])
def api_info():
    result = {
        "code": 0,
        "roles": info(),
    }

    return jsonify(result)


def get_all_roles():
    supported_roles = set(["read", "readWrite", "dbAdmin"])
    users_dict = {}

    for db_id in mongo_idm.helpers.get_databases():
        for user in mongo_idm.db.mongo_users_info(db_id)["users"]:
            if not user.get("customData", {}).get("idm", False):
                continue

            roles = [role["role"] for role in user["roles"]]

            users_dict[user["user"]] = users_dict.get(user["user"], [])

            users_dict[user["user"]] += [
                {db_id: role} for role in roles if role in supported_roles
            ]

    result = [
        {"login": k, "roles": v} for k, v in users_dict.items()
    ]

    return result


@api.route("/get-all-roles/", methods=["GET"])
def api_get_all_roles():
    result = {
        "code": 0,
        "users": get_all_roles(),
    }
    return jsonify(result)


def add_role(db_id, user, role):
    password = mongo_idm.helpers.generate_password()
    mongo_idm.helpers.create_vault_secret(db_id, user, password)
    mongo_idm.db.mongo_create_user_if_not_exists(
        db_id, user, password
    )
    mongo_idm.db.mongo_grant_role_to_user_if_not_granted(db_id, user, role)


@api.route("/add-role/", methods=["POST"])
def api_add_role():
    user = request.form["login"]
    idm_role = json.loads(request.form["role"])
    db_id = idm_role["database"]
    role = idm_role["role"]

    add_role(db_id, user, role)

    result = {"code": 0}

    return jsonify(result)


def remove_role(db_id, user, role, fired):
    mongo_idm.db.mongo_revoke_role_from_user_if_exists(db_id, user, role)

    if fired:
        mongo_idm.db.mongo_drop_user_if_exists(db_id, user)
    else:
        mongo_idm.db.mongo_drop_user_if_has_no_roles(db_id, user)


@api.route("/remove-role/", methods=["POST"])
def api_remove_role():
    user = request.form["login"]
    idm_role = json.loads(request.form["role"])
    db_id = idm_role["database"]
    role = idm_role["role"]
    fired = True if int(request.form.get("fired", 0)) == 1 else False

    remove_role(db_id, user, role, fired)

    result = {"code": 0}

    return jsonify(result)


@api.route("/ping")
def ping():
    result = {"code": 0}

    return jsonify(result)
