import os
import logging
import pymongo


class MongoDBCached(object):
    def __init__(self):
        self._client_db_id_map = {}

    def _check_client_alive(self, db_id):
        db = self._client_db_id_map[db_id][self._get_mongo_db(db_id)]
        logging.info("Check database connection")

        try:
            if "ping" not in db.collection_names():
                db.create_collection("ping", capped=True, size=128, max=128)

            db["ping"].insert({})
            logging.info("Database connection successful")
        except:
            logging.exception("Can't insert in the collection ping")
            return False

        return True

    def _get_mongo_host(self, db_id):
        return str(os.environ[db_id + '_MONGO_HOST'])

    def _get_mongo_port(self, db_id):
        return int(os.environ[db_id + '_MONGO_PORT'])

    def _get_mongo_timeout(self, db_id):
        return int(os.environ[db_id + '_MONGO_CONNECT_TIMEOUT_MS'])

    def _get_mongo_db(self, db_id):
        return str(os.environ[db_id + '_MONGO_DB'])

    def _get_mongo_user(self, db_id):
        return str(os.environ[db_id + '_MONGO_USER'])

    def _get_mongo_pwd(self, db_id):
        return str(os.environ[db_id + '_MONGO_PWD'])

    def _get_mongo_client(self, db_id):
        if db_id in self._client_db_id_map and \
           self._check_client_alive(db_id):
            return self._client_db_id_map[db_id]
        else:
            client = pymongo.MongoClient(
                host=self._get_mongo_host(db_id),
                port=self._get_mongo_port(db_id),
                connectTimeoutMS=self._get_mongo_timeout(db_id),
            )

            client[self._get_mongo_db(db_id)].authenticate(
                self._get_mongo_user(db_id),
                self._get_mongo_pwd(db_id)
            )

            self._client_db_id_map[db_id] = client

            return self._client_db_id_map[db_id]

    def get_mongo_db(self, db_id):
        logging.info("Acquiring connection to db {}".format(db_id))
        return self._get_mongo_client(db_id)[self._get_mongo_db(db_id)]


def mongo_user_info(db_id, user):
    db = MongoDBCached().get_mongo_db(db_id)

    user_info = db.command("usersInfo", {"user": user, "db": db.name})
    logging.info("Got users info from mongo: {}".format(user_info))

    return user_info


def mongo_users_info(db_id):
    db = MongoDBCached().get_mongo_db(db_id)

    users_info = db.command("usersInfo")
    logging.info("Got users info from mongo: {}".format(users_info))

    return users_info


def mongo_create_user_if_not_exists(db_id, user, password):
    db = MongoDBCached().get_mongo_db(db_id)
    user_info = mongo_user_info(db_id, user)

    if not user_info["users"]:
        logging.info("Creating user {}".format(user))
        db = MongoDBCached().get_mongo_db(db_id)
        db.command(
            "createUser",
            user,
            pwd=password,
            roles=["changeOwnPassword"],
            customData={"idm": True},
        )


def mongo_grant_role_to_user_if_not_granted(db_id, user, role):
    db = MongoDBCached().get_mongo_db(db_id)
    user_info = mongo_user_info(db_id, user)

    if {"role": role, "db": db.name} not in user_info["users"][0]["roles"]:
        logging.info(
            "Granting role {} to user {} at db {}".format(role, user, db_id)
        )
        db.command(
            "grantRolesToUser", user,
            roles=[{"role": role, "db": db.name}],
        )


def mongo_revoke_role_from_user_if_exists(db_id, user, role):
    db = MongoDBCached().get_mongo_db(db_id)
    user_info = mongo_user_info(db_id, user)

    if user_info["users"]:
        if {"db": db.name, "role": role} in user_info["users"][0]["roles"]:
            logging.info(
                "Revoking role {} from user {} at db {}".format(
                    role, user, db_id
                )
            )
            db.command(
                "revokeRolesFromUser", user,
                roles=[{"role": role, "db": db.name}],
            )


def mongo_drop_user_if_exists(db_id, user):
    db = MongoDBCached().get_mongo_db(db_id)
    user_info = mongo_user_info(db_id, user)

    if user_info["users"]:
        logging.info("Dropping user {} at db {}".format(user, db_id))
        db.command("dropUser", user)


def mongo_drop_user_if_has_no_roles(db_id, user):
    user_info = mongo_user_info(db_id, user)

    if user_info["users"] and \
       len(user_info["users"][0]["roles"]) == 1 and \
       user_info["users"][0]["roles"][0]["role"] == "changeOwnPassword":
        logging.info("Dropping user {} at db {}".format(user, db_id))
        mongo_drop_user_if_exists(db_id, user)
