import os
import random
import string
import logging
from library.python.vault_client.instances import Production as VaultClient


def generate_password(pass_len=64):
    if pass_len < 32:
        raise Exception("Too short password")
    return "".join(
        [random.choice(string.letters + string.digits) for x in range(pass_len)]
    )


def get_databases():
    dbs = os.environ['DATABASES'].split(',')
    logging.info("Acquired databases: {}".format(dbs))

    return dbs


def create_vault_secret(db_id, user, password):
    vault_client = VaultClient(
        authorization="Oauth {}".format(os.environ["YAV_OAUTH_TOKEN"])
    )

    prefix = "PAYSYS_MONGO_IDM"

    tags = "{},{},{}".format(prefix, db_id, user)
    name = "{}.{}.{}".format(prefix, db_id, user)
    version_ttl = 90 * 86400
    secret_comment = "Autogenerated secret by mongo-idm service"
    version_comment = "This version will expire in 90 days. " \
                      "You must request the role in IDM again to restore access."
    secret_id = None

    secrets = vault_client.list_secrets(tags=tags, yours=True)
    logging.info("Got secrets from vault with tags {}: {}".format(tags, secrets))

    if len(secrets) > 1:
        raise "There should be only one secret with this set of params"
    elif len(secrets) == 0:
        secret_id = vault_client.create_secret(name, secret_comment, tags)
        logging.info("Created new yav secret with id {}".format(secret_id))
    else:
        secret_id = secrets[0]["uuid"]
        result = vault_client.update_secret(
            secret_id, name, secret_comment, tags
        )
        if not result:
            raise Exception("Can't update secret with id {}".format(secret_id))
        logging.info("Updated yav secret with id {}".format(secret_id))

    vault_client.add_user_role_to_secret(
        secret_uuid=secret_id,
        role="READER",
        login=user,
    )
    logging.info(
        "Added access to secret {} for user {}".format(secret_id, user)
    )

    version_id = vault_client.create_secret_version(
        secret_uuid=secret_id,
        value={"password": password},
        ttl=version_ttl,
        comment=version_comment,
    )
    logging.info("Created version {} of yav secret".format(version_id))

    version = vault_client.get_version(version_id, packed_value=True)

    if version["value"] != {"password": password}:
        raise Exception(
            "Version {} of secret {} wasn't added properly".format(
                secret_id, version_id
            )
        )

    logging.info("Checked version {} of yav secret".format(version_id))
