import requests
import argparse
import logging
import sys
import json
import jinja2


class BunkerApi:
    def __init__(self, environment: str, notify_name: str):
        self.environment = environment
        self.notify_name = notify_name

    @staticmethod
    def api_url_by_env_notify_name(environment: str, notify_name: str):
        if environment == 'prod':
            host = 'http://bunker-api.yandex.net/v1/cat?node='
        else:
            host = 'http://bunker-api-dot.yandex.net/v1/cat?node='

        url = host + '{}&version=latest'.format(notify_name)
        return url

    @staticmethod
    def api_url_by_env(environment: str):
        if environment == 'prod':
            host = 'http://bunker-api.yandex.net/v1/ls?node=/balance-notifier/prod/&version=latest'
        else:
            host = 'http://bunker-api-dot.yandex.net/v1/ls?node=/balance-notifier/test/&version=latest'
        return host


def call_api(http_handle: str) -> dict:
    response = requests.get(http_handle)

    try:
        response.raise_for_status()
    except requests.HTTPError:
        logging.warning(f'Response body: {response.text}')
        raise

    return response.json()


def json_result_write_out_file(input_json_data: list, output_file: str) -> None:
    with open(output_file, 'a+') as f:
        if input_json_data:
            json.dump(input_json_data, f, ensure_ascii=False, indent=4)
        else:
            json.dump('', f, ensure_ascii=False, indent=4)


def parser() -> argparse:
    parser_ = argparse.ArgumentParser(description="This program must be run from the root of Arcadia.")
    parser_.add_argument('-env', '--environment', help='Environment', default='prod', choices=['test', 'prod'])
    parser_.add_argument('-nodes', '--nodes',
                         help='The node (mailing), comma separated from bunker three', required=True)
    parser_.add_argument('-of', '--out_file', help='Output file name', default='output_json')
    parser_.add_argument('-ll', '--log_level', help='Log level Bunker', default='info',
                         choices=['debug', 'info', 'warning', 'error', 'critical'])
    parser_.add_argument('-tp', '--template_params', help='template params for query', default='{}')

    return parser_.parse_args()


def logging_settings(args) -> None:
    logging.basicConfig(
        level=args.log_level.upper(),
        stream=sys.stdout
    )


def get_bunker_config(environment: str, nodes: str, out_file: str, template_params: dict) -> None:
    """
    Getting config from bunker for exact notify email name
    :param environment:
    :param nodes:
    :param out_file:
    :param template_params:
    :return:
    """
    bunker_url = BunkerApi.api_url_by_env(environment)
    logging.info(f'Getting list of nodes. Request: {bunker_url}')
    notify_nodes = call_api(bunker_url)
    processing_nodes: list = nodes.split(',')

    for notify in notify_nodes:
        if notify['name'] in processing_nodes:
            request_notify = BunkerApi.api_url_by_env_notify_name(environment, notify['fullName'])

            logging.info(f'Getting exact notify parameters. Request: {request_notify}')
            notify_params = call_api(request_notify)

            if template_params is not None:
                for params in notify_params:
                    if 'query' in params:
                        params['query'] = jinja2.Template(params['query']).render(template_params)
        else:
            continue

        logging.info('Writing result to output json file')
        json_result_write_out_file(list(notify_params), out_file)


def main() -> None:
    args = parser()
    logging_settings(args)

    get_bunker_config(args.environment, args.nodes, args.out_file, json.loads(args.template_params))

if __name__ == '__main__':
    main()
