import requests
import sys
import logging
import argparse
import tvmauth
from yql.api.v1.client import YqlClient
from typing import List, Dict

BALANCE_NOTIFIER_TEST_ML = 'balance-notifier-test@yandex-team.ru'


def check_message_status(notify_url: str, message_id: str, tvm_ticket: str) -> int:
    """
    Checking delivery status
    :param notify_url:
    :param message_id:
    :param tvm_ticket:
    :return:
    """
    notify_check = notify_url+f'/status?message_id={message_id}'

    response = requests.get(
        notify_check,
        headers={'Content-Type': 'application/json',
                 'X-Ya-Service-Ticket': tvm_ticket}
    )

    try:
        response.raise_for_status()
    except requests.HTTPError:
        logging.warning(f'Response body: {response.text}')
        raise

    logging.info(f'Message status: {response.text}')

    status = response.json()
    return status['code']


def sender(list_notify: List[Dict], env: str, self_secret: str, self_tvm_id: int, sender_tvm_id: int) -> None:
    """
    Prepare and send emails through sender
    :param list_notify:
    :param env:
    :param self_secret:
    :param self_tvm_id:
    :param sender_tvm_id:
    :return:
    """
    if env == 'test':
        list_notify = [
            {
                **x,
                'email': BALANCE_NOTIFIER_TEST_ML if 'email' in x and x['email'] != '' else '',
                'to': BALANCE_NOTIFIER_TEST_ML if 'to' in x and x['to'] != '' else '',
                'cc': BALANCE_NOTIFIER_TEST_ML if 'cc' in x and x['cc'] != '' else '',
                'bcc': BALANCE_NOTIFIER_TEST_ML if 'bcc' in x and x['bcc'] != '' else ''
            }
            for x in list_notify
        ]

    tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
        self_tvm_id=self_tvm_id,
        self_secret=self_secret,
        dsts={"sender": sender_tvm_id}
    ))
    tvm_ticket = tvm_client.get_service_ticket_for("sender")

    for notify in list_notify:

        logging.info('Checking message status')
        status_code = check_message_status(notify['notify_url'], notify['message_id'], tvm_ticket)

        if status_code == 250:
            logging.info("The email has already been sent, we will not repeat it")
            continue

        if not notify['to']:
            notify['to'] = notify['email']

        to_ = notify['to'].split(',')
        list_to: list = [{'email': x} for x in to_]

        dict_sender_param: dict = {"args": {**notify},
                                   "to": list_to,
                                   "async": "true",
                                   "message_id": notify['message_id']}

        if notify['cc']:
            cc_ = notify['cc'].split(',')
            dict_sender_param['cc'] = [{'email': x} for x in cc_]

        if notify['bcc']:
            bcc_ = notify['bcc'].split(',')
            dict_sender_param['bcc'] = [{'email': x} for x in bcc_]

        notify_send = notify['notify_url']+'/send'

        response = requests.post(
            notify_send,
            json=dict_sender_param,
            headers={'Content-Type': 'application/json',
                     'X-Ya-Service-Ticket': tvm_ticket}
        )

        try:
            response.raise_for_status()
        except requests.HTTPError:
            logging.warning(f'Response body: {response.text}')
            raise

        logging.info(f'Send email: {response.text}')


def parser() -> argparse:
    parser_ = argparse.ArgumentParser(description="This program must be run from the root of Arcadia.")
    parser_.add_argument('-env', '--environment', help='Environment', default='prod', choices=['test', 'prod'])
    parser_.add_argument('-ftn', '--file_with_table_name', help='Input YT table with managers to notify', required=True)
    parser_.add_argument('-ttf', '--tvm_token_file', help='TVM token file', required=True)
    parser_.add_argument('-ytf', '--yql_token_file', help='YQL token file', required=True)
    parser_.add_argument('-cluster', '--cluster', help='YT cluster', default='hahn', choices=['hahn', 'arnold'])
    parser_.add_argument('-self_tvm_id', '--self_tvm_id', help='Balance notifier TVM ID', required=True, type=int)
    parser_.add_argument('-sender_tvm_id', '--sender_tvm_id', help='Sender TVM ID', required=True, type=int)
    parser_.add_argument('-ll', '--log_level', help='Log level Sender', default='info',
                         choices=['debug', 'info', 'warning', 'error', 'critical'])

    return parser_.parse_args()


def logging_settings(args) -> None:
    logging.basicConfig(
        level=args.log_level.upper(),
        stream=sys.stdout
    )


def read_table_with_managers_to_notify(table_path: str, yql_token: str, cluster: str) -> list[Dict]:
    yql_response = YqlClient(db=cluster, token=yql_token).query(f'select * from `{table_path}`').run()
    logging.info(f"YQL shared link: {yql_response.share_url}")
    dataframe_result_from_yt = yql_response.get_results().dataframe

    if dataframe_result_from_yt.empty:
        logging.info(f'The result YT table {table_path} with managers to notify is empty, nothing to do.')
        sys.exit(0)

    wrapped_emails_uuid = dataframe_result_from_yt.to_dict('records')
    emails_to_notify = unwrap_emails_uuid(wrapped_emails_uuid)
    return emails_to_notify


def unwrap_emails_uuid(wrapped_emails: list[Dict]) -> list[Dict]:
    """
    Getting rid of unnecessary nesting
    :param wrapped_emails:
    :return:
    """
    return [
        {
            **x['message_to_send'],
            'message_id': x['message_id']
        }
        for x in wrapped_emails
    ]


def main() -> None:
    args = parser()
    logging_settings(args)

    with open(args.tvm_token_file, 'r') as f:
        tvm_token_file = f.read()

    with open(args.yql_token_file, 'r') as f:
        yql_token_file = f.read()

    with open(args.file_with_table_name, 'r') as f:
        file_with_table_name = f.read()

    if len(file_with_table_name) == 0:
        logging.info('Table name is empty. Nothing to read.')
        return

    logging.info('Reading table with managers to notify')
    list_notify = read_table_with_managers_to_notify(file_with_table_name.rstrip(),
                                                     yql_token_file.rstrip(),
                                                     args.cluster)

    logging.info('Calling Sender API to send emails')
    sender(list_notify, args.environment, tvm_token_file.rstrip(), args.self_tvm_id, args.sender_tvm_id)


if __name__ == '__main__':
    main()
