#!/usr/bin/env python

import socket
import argparse
from psadmin import base

name = base.get_name()
check_name = "haproxy"
log = base.log_file(name)

def parse_args():
    parser = argparse.ArgumentParser(
                 description='Check if all haproxy backends are up and running'
             )
    parser.add_argument(
        '--check-backends-up', dest='check_backends_up',
        action='store_true', required=False,
        help='If backend is in UP state - this is ok' + \
             ' (doesn\'t matter how many of servers up)'
    )
    return base.finalize_argparser(parser)


def haproxy_command(cmd):
    # Connect to haproxy socket
    haproxy_socket = socket.socket(socket.AF_UNIX)
    haproxy_socket.connect("/var/run/haproxy.socket")

    # Put show stats in it
    haproxy_socket.send(cmd + "\n")

    # Read all responce
    out = ""
    while 1:
        out_part = haproxy_socket.recv(4096)
        if out_part:
            out += out_part
        else:
            break

    log.debug(out)
    return out


def main():
    args = parse_args()

    # Haproxy stat csv output map
    haproxy_stats = "pxname,svname,qcur,qmax,scur,smax,slim,stot,bin,bout,dreq,dresp,ereq,econ,eresp,wretr,wredis,status,weight,act,bck,chkfail,chkdown,lastchg,downtime,qlimit,pid,iid,sid,throttle,lbtot,tracked,type,rate,rate_lim,rate_max,check_status,check_code,check_duration,hrsp_1xx,hrsp_2xx,hrsp_3xx,hrsp_4xx,hrsp_5xx,hrsp_other,hanafail,req_rate,req_rate_max,req_tot,cli_abrt,srv_abrt"
    haproxy_stat_map = {}
    for i in enumerate(haproxy_stats.split(",")):
        haproxy_stat_map[i[1]] = i[0]

    haproxy_stat = haproxy_command("show stat")

    # Check some params (like status) at each line
    for line in haproxy_stat.split("\n"):
        if (len(line) and line[0] == "#") or not len(line):
            continue
        else:
            line_s = line.split(",")
            if args.check_backends_up and \
               line_s[haproxy_stat_map["svname"]] != 'BACKEND':
                continue
            if line_s[haproxy_stat_map["status"]] != "OPEN" and \
               not line_s[haproxy_stat_map["status"]].startswith('UP'):
                base.monrun_output(check_name, 2, "{}/{} is {}".format(
                        line_s[haproxy_stat_map["pxname"]],
                        line_s[haproxy_stat_map["svname"]],
                        line_s[haproxy_stat_map["status"]]
                    )
                )
                return

    # Check level on socket
    # config should look like this:
    # stats socket /var/run/haproxy.socket mode 666 level user
    if haproxy_command("show errors") != "Permission denied\n\n":
        base.monrun_output(
            check_name, 2,
            "Too high priv level for haproxy socket (set user)"
        )
        return

    base.monrun_output(check_name, 0, "Ok")


if __name__ == "__main__":
    main()

