#!/usr/bin/env python2
from __future__ import division
from datetime import datetime
import requests
import json
import logging
import argparse
import os
import sys
import fcntl

from psadmin import base

name = base.get_name()
log = logging.getLogger(__name__)
handler = logging.StreamHandler(sys.stdout)
formatter = logging.Formatter(
    "[%(filename)s:%(lineno)s - %(funcName)10s() ] %(message)s"
)
handler.setFormatter(formatter)
log.setLevel(logging.INFO)
log.addHandler(handler)


class Lock(object):
    def __init__(self, lock_file):
        self.lock_file = lock_file
        self.lock_fd = None

    def __fileno(self):
        lock = open(self.lock_file, 'r')
        return lock

    def __enter__(self):
        try:
            self.lock_fd = self.__fileno()
            fcntl.flock(self.lock_fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
            return self
        except IOError as e:
            log.exception('Could not get lock on {}: {}'.format(
                self.lock_file,
                str(e))
            )
            raise

    def __exit__(self, exc_type, exc_val, exc_tb):
        fcntl.flock(self.lock_fd, fcntl.LOCK_UN)
        if exc_val:
            raise


def parse_args():
    parser = argparse.ArgumentParser(
        description="Monitoring documents queue in uzedo"
    )

    parser.add_argument(
        '--threshold_time',
        default=3600,
        type=int
    )
    return base.finalize_argparser(parser)


def do_request(
    method,
    path,
    host='[::1]',
    port=8080,
    data=None,
    headers=None,
    payload=None
):
    url = 'http://{}:{}/{}'.format(host, port, path)
    default_headers = {
        'Content-Type': 'application/json;charset=UTF-8'
    }
    if isinstance(headers, dict):
        default_headers.update(headers)

    if method == 'GET':
        r = requests.get(url, headers=default_headers, params=payload)

    elif method == 'POST':
        r = requests.post(
            url,
            data=json.dumps(data),
            headers=default_headers
        )
    r.raise_for_status()

    return r


def get_session_id(username, password, path):
    payload = {
        "username": username,
        "password": password
    }

    r = do_request('POST', path['login'], data=payload)
    session = {'Cookie': r.headers['Set-Cookie']}
    return session


def get_documents(session, path, types=['NotSent', 'Error']):
    result = {}
    for doc_type in types:
        result[doc_type] = []
        params = {
            'basketId': doc_type,
            'includeCount': False,
            'page': 0,
            'size': 1100,
        }
        while True:
            r = do_request(
                'GET',
                path['doc_list'],
                payload=params,
                headers=session
            )

            documents = r.json()
            for document in documents:
                result[doc_type].append(document["modified"])
            params['page'] += 1
            if len(documents) < params['size']:
                break

    return result


def check_cache_file(mon_file):
    status = 0
    message = ''
    s = os.stat(mon_file)
    delta = datetime.now() - datetime.fromtimestamp(s.st_mtime)
    # check if cache not updated > 4 hours cache is stale
    if delta.total_seconds() > 14400:
        status = 2
        message = '{} is too old'.format(mon_file)
        data = 'PASSIVE-CHECK:{};{};{}'.format(
            'uzedo_docs_in_queue',
            status,
            message
        )
        with open(mon_file, 'w') as f:
            f.write(data)

def daemonize(mon_file):

        # read monitoring data
        # first fork + detach from session
        pid = os.fork()
        if pid > 0:
            check_cache_file(mon_file)
            with open(mon_file, 'r') as f:
                print(f.readline())
            sys.exit(0)
        # change env for child
        os.setsid()
        os.chdir("/")
        os.umask(0)

        # second fork for removing session leader
        if pid > 0:
            sys.exit(0)

        fd = os.open("/dev/null", os.O_RDWR)
        os.dup2(fd, 0)
        os.dup2(fd, 1)
        os.dup2(fd, 2)


def refresh_in_background_monitoring_data(
    lock_file,
    mon_file,
    user,
    password,
    path,
    threshold_time=3600
):
    with Lock(lock_file):
        session = get_session_id(user, password, path)
        documents = get_documents(session, path)

        message = {}
        status = 0
        cur_time = datetime.now()
        for doc_type in documents.keys():
            for document in documents[doc_type]:
                doc_modify = datetime.fromtimestamp(document / 1000)
                delta = cur_time - doc_modify
                if delta.total_seconds() > threshold_time:
                    status = 2
                    try:
                        message[doc_type] += 1
                    except KeyError:
                        message[doc_type] = 1

        data = 'PASSIVE-CHECK:{};{};{}'.format(
            'uzedo_docs_in_queue',
            status,
            'Docs stale in queue: {} more than {} sec'.format(
                message,
                threshold_time
            )
        )
        # write monitoring data to a file
        with open(mon_file, 'w') as f:
            f.write(data)


def main():
    args = parse_args()
    conf = '/etc/yandex/paysys/juggler/checks/uzedo_docs_in_queue/config.json'
    lock_file = '/tmp/uzedo_docs_in_queue.lock'
    mon_file = '/tmp/uzedo_docs_in_queue.mon'
    status = 0
    check = 'uzedo_docs_in_queue'
    try:
        with open(conf, 'r') as config_file:
            config = json.load(config_file)
    except IOError as e:
        status = 2
        message = '{} {}'.format(e.filename, e.strerror)
        return base.monrun_output(
            check,
            status,
            '{}'.format(message)
        )
    user = config['username']
    password = config['password']
    path = {
        'login': 'client_api/login',
        'doc_list': 'client_api/document/list',
    }

    for f in [lock_file, mon_file]:
        try:
            os.mknod(f)
        except OSError:
            log.debug('{} file already exists, skip creation...'.format(f))

    daemonize(mon_file)
    refresh_in_background_monitoring_data(
        lock_file,
        mon_file,
        user,
        password,
        path,
        threshold_time=args.threshold_time
    )


if __name__ == '__main__':
    main()
