import jinja2
from loguru import logger
import argparse
import paramiko
import re
import cx_Oracle
import tabulate
import json
from functools import reduce

tabulate.PRESERVE_WHITESPACE = True


def yes_no(message: str, default: str = 'yes', suffix: str = ' ') -> bool:
    """
    Prompt user to answer yes or no. Return True if the default is chosen, otherwise False.
    :param message:
    :param default:
    :param suffix:
    :return:
    """
    if default == 'yes':
        yesno_prompt = '[Y/n]'
    elif default == 'no':
        yesno_prompt = '[y/N]'
    else:
        raise ValueError("Default must be 'yes' or 'no'.")

    if message != '':
        prompt_text = f"{message} {yesno_prompt}{suffix}"
    else:
        prompt_text = f"{yesno_prompt}{suffix}"

    while True:
        response = input(prompt_text).strip()
        if response == '':
            return True
        else:
            if re.match('^(y)(es)?$', response, re.IGNORECASE):
                if default == 'yes':
                    return True
                else:
                    return False
            elif re.match('^(n)(o)?$', response, re.IGNORECASE):
                if default == 'no':
                    return True
                else:
                    return False


def is_in(regex: str, text: str) -> bool:
    """
    :param regex:
    :param text:
    :return:
    """
    pattern = re.compile(regex)
    if pattern.search(text) is not None:
        return True
    else:
        return False


def read_ora_pass_file() -> str:
    """
    Reading oracle password file
    :return:
    """
    with open(config.ORA_PASS_FILE, "r") as sys_pass_open:
        sys_pass_full = sys_pass_open.read()
        sys_pass = sys_pass_full.split('\npassword=')
        sys_pass = sys_pass[1].split('\n')

    if sys_pass:
        return sys_pass[0]
    else:
        return ''


def load_projects() -> list:
    """
    Reading projects file
    :return:
    """
    logger.info('')
    with open(config.PROJECTS, "r") as f:
        projects = json.load(f)

    if projects:
        return projects['projects']
    else:
        return []


def load_steps() -> list:
    """
    Reading steps file
    :return:
    """
    logger.info('')
    with open(config.STEPS, "r") as f:
        steps = json.load(f)

    if steps:
        return steps['steps']
    else:
        return []


def connection(dsn_tns: str) -> cx_Oracle.connect:
    """
    Making connection
    :param dsn_tns:
    :return:
    """
    sys_pass = read_ora_pass_file()
    try:
        con = cx_Oracle.connect('SYS', sys_pass, dsn=dsn_tns, mode=cx_Oracle.SYSDBA, encoding='utf-8', nencoding='utf-8')
        return con
    except Exception as e:
        logger.error(f"Can't connect to: {dsn_tns}")
        logger.debug(e)
        return cx_Oracle.connect()


def checking_db_role(host: dict) -> str:
    """
    Checking Database role
    :param host:
    :return:
    """
    try:
        dsn_tns = cx_Oracle.makedsn(host['fqdn'], '1521', sid=host['sid'])
        conn = connection(dsn_tns)
        cur = conn.cursor()
        cur.execute("select database_role from v$database")
        db_role = cur.fetchone()
        return db_role[0]
    except Exception:
        return 'EXCEPTION! EXCEPTION! EXCEPTION!'


def preparing_hosts(datacenters: list, env: str, prj: str) -> list:
    """
    Preparing the hosts which will involve to switchover
    :param datacenters:
    :param env:
    :param prj:
    :return:
    """
    projects = load_projects()

    if not projects:
        logger.error("No any projects loaded")
        exit(0)

    hosts = []
    project_hosts = []

    logger.info('Identifying hosts involved to switchover process')
    for dc in datacenters:
        for p in projects:
            if dc == p['dc'] and env == p['env'] and prj == p['project']:
                hosts.append(p)
            else:
                continue

    if not hosts:
        logger.error('No dc or env or prj were given')

    logger.info('Mapping the PRIMARY and STANDBY roles')
    for host in hosts:
        db_role = checking_db_role(host)
        host['role'] = db_role
        project_hosts.append(host)

    header = project_hosts[0].keys()
    rows = [x.values() for x in project_hosts]
    logger.info('\n{0}'.format(tabulate.tabulate(rows, header)))

    return project_hosts


def preparing_command(template_step: dict, project_hosts: list) -> list:
    """
    Preparing the command with fqdn where it should be execute
    :param template_step:
    :param project_hosts:
    :return:
    """
    ora_password = read_ora_pass_file()
    result_hosts_to_execute = []

    if not ora_password:
        logger.error("Oracle password is empty")
        exit(0)

    for host in project_hosts:
        if (template_step['role'] == host['role'] or template_step['role'] == 'BOTH') \
                and template_step['id'] in host['steps']:
            host['pass'] = ora_password
            command = jinja2.Template(template_step['command']).render(host)
        else:
            continue

        result_hosts_to_execute.append({'fqdn': host['fqdn'],
                                        'oracle_home': host['oracle_home'],
                                        'sid': host['sid'],
                                        'command': command,
                                        'id': template_step['id'],
                                        'container': host['container'],
                                        'tns_port': host['tns_port']})

    return result_hosts_to_execute


def performing_dry_run(exec_block: list) -> None:
    """
    Showing exact command how it will be execute, but not executing it
    :param exec_block:
    :return:
    """
    for block in exec_block:
        logger.info(f"STEP_ID: {block['id']} {block['fqdn']}: {block['command']}")


def performing_manual(exec_block: list) -> None:
    """
    Executing the command with user confirmation
    :param exec_block:
    :return:
    """
    for block in exec_block:
        logger.warning(f"Do you wanna execute {block['fqdn']}: {block['command']}")
        if yes_no('Continue?'):
            performing_command(block)
        else:
            logger.debug('Skipping current step')
            continue


def performing_without_questions(exec_block: list, step_id: int) -> None:
    """
    Executing the commands in automatic mode
    :param exec_block:
    :param step_id:
    :return:
    """
    for block in exec_block:
        if step_id:
            if step_id == block['id']:
                logger.info(f"Executing step ID: {block['id']}")
                performing_command(block)
        else:
            logger.info(f"Executing step ID: {block['id']}")
            performing_command(block)


def preparing_steps(project_hosts: list, step_id: int, dry_run: bool, manual: bool) -> None:
    """
    Main function for steps. Listing all commands which will be executed, and calling them with different modes
    :param project_hosts:
    :param step_id:
    :param dry_run:
    :param manual:
    :return:
    """
    template_steps = load_steps()

    if not template_steps:
        logger.error("No any steps loaded")
        exit(0)

    project_steps = sorted(set(reduce(lambda x, y: x + y, [x_['steps'] for x_ in project_hosts])))

    header = template_steps[0].keys()
    rows = [x.values() for x in template_steps if x['id'] in project_steps]
    logger.info('\n{0}'.format(tabulate.tabulate(rows, header)))

    for project_step in project_steps:
        for template_step in template_steps:
            if project_step == template_step['id']:
                if template_step['enabled']:
                    exec_block = preparing_command(template_step, project_hosts)
                    if dry_run:
                        performing_dry_run(exec_block)
                    elif manual:
                        performing_manual(exec_block)
                    else:
                        performing_without_questions(exec_block, step_id)
                else:
                    logger.error(f"The step ID: {template_step['id']} is not enabled, skipping")


def switchover(datacenters: list, step_id: int, env: str, prj: str, dry_run: bool, manual: bool) -> None:
    """
    Main function for switchover process
    :param datacenters:
    :param step_id:
    :param env:
    :param prj:
    :param dry_run:
    :param manual:
    :return:
    """
    if dry_run:
        logger.warning('Switchover process started as a dry run!')
    elif manual:
        logger.warning('Switchover process started with manual option')
    elif step_id:
        logger.warning(f'Switchover process started for only one step id: {step_id}')
    else:
        logger.warning('Switchover process started')

    if env and prj:
        project_hosts = preparing_hosts(datacenters, env, prj)
        preparing_steps(project_hosts, step_id, dry_run, manual)
    else:
        logger.error('ENV or PRJ cannot be empty')

    logger.warning('Switchover process finished')


def exec_sql(block: dict) -> None:
    """
    Execute SQL
    :param block:
    :return:
    """
    result = []

    if is_in(config.CONTAINER_TEMPLATE_CMD, block['command']):
        dsn_tns = cx_Oracle.makedsn(block['fqdn'], block['tns_port'], service_name=block['container'])
    else:
        dsn_tns = cx_Oracle.makedsn(block['fqdn'], block['tns_port'], sid=block['sid'])

    conn = connection(dsn_tns)
    cursor = conn.cursor()

    try:
        cursor.execute(block['command'])
        result.append(cursor.fetchall())
    except cx_Oracle.InterfaceError:
        logger.info(f"{block['fqdn']}: {block['command']} ")
        logger.info('DDL executed')
        conn.commit()
    except cx_Oracle.DatabaseError as e:
        logger.error(e)

    if result:
        header = [row[0] for row in cursor.description]
        logger.info('\n{0}'.format(tabulate.tabulate(result[0], header)))
    else:
        pass


def exec_ssh(block: dict) -> None:
    """
    SSH connect, and executing the command
    :param block:
    :return:
    """
    stdout_buff = ''
    stderr_buff = ''
    try:
        client = paramiko.SSHClient()
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        client.connect(hostname=block['fqdn'], username=config.USER, key_filename=config.PUBLIC_KEY)
        stdin, stdout, stderr = client.exec_command(block['command'])

        logger.info(f"{block['fqdn']}: {block['command']} ")

        for line in stdout.readlines():
            stdout_buff += line

        for line in stderr.readlines():
            stderr_buff += line

        if stderr_buff:
            logger.info(stderr_buff.strip('\x00'))

        if stdout_buff:
            logger.info(stdout_buff.strip('\x00'))
    except Exception as e:
        logger.debug(e)
        exit(0)


def performing_command(block: dict) -> None:
    """
    Defining the command type through templates
    :param block:
    :return:
    """
    if is_in(config.SQL_TEMPLATE_CMD, block['command']):
        logger.info("The command found in SQL template")
        exec_sql(block)
    elif is_in(config.ORA_TEMPLATE_CMD, block['command']):
        logger.info("The command found in ORA template")
        exec_ssh(block)
    elif is_in(config.OS_TEMPLATE_CMD, block['command']):
        logger.info("The command found in OS template")
        exec_ssh(block)
    else:
        logger.error(f"Can't find template for command: {block['command']}")


def show_steps() -> None:
    """
    Print steps
    :return:
    """
    steps = load_steps()
    header = steps[0].keys()
    rows = [x.values() for x in steps]
    logger.info('\n{0}'.format(tabulate.tabulate(rows, header)))
    exit(0)


def parser() -> argparse:
    """
    Args parse
    :return:
    """
    parser_ = argparse.ArgumentParser()

    a_show_steps = parser_.add_argument_group("Show steps")
    a_switchover = parser_.add_argument_group("Switchover")

    arg_show_steps(a_show_steps)
    arg_switchover(a_switchover)

    parser_.add_argument('-single', help='Single step id', type=int)
    parser_.add_argument('-dry_run', default=False, help='Dry Run', action='store_true')
    parser_.add_argument('-manual', default=False, help='Manual Run', action='store_true')
    return parser_.parse_args()


def arg_show_steps(group: argparse) -> None:
    group.add_argument('-show_steps', help='Showing the all steps', action='store_true')


def arg_switchover(group: argparse) -> None:
    group.add_argument('-switchover', help='Switchover database', nargs=2, metavar=('from_dc', 'to_dc'))
    group.add_argument('-env', help='Environment')
    group.add_argument('-prj', help='Project')


def main() -> None:
    """
    Main
    :return:
    """
    logger.add("log/dbctl_{time}.log", rotation="10 MB")
    args = parser()

    if args.show_steps:
        show_steps()

    switchover(args.switchover, args.single, args.env, args.prj, args.dry_run, args.manual)


if __name__ == '__main__':
    try:
        from paysys.sre.oracle.dbctl import config
    except ImportError:
        import config

    main()
