#!/usr/bin/env python
import pymongo
import time
import logging
import ConfigParser
import argparse
import socket
from psadmin import solomon

log = logging.getLogger()
sh = logging.StreamHandler()
log.addHandler(sh)
log.setLevel(logging.INFO)


def parse_args():
    parser = argparse.ArgumentParser(
        description='Checks various params of mongodb and '
                    'reports status to stdout/graphite'
    )
    parser.add_argument('-v', '--verbose', dest='verbose', action='store_true')
    parser.add_argument(
        '--host',
        action='store',
        type=str,
        dest='host',
        default='127.0.0.1',
        help='The hostname you want to connect to'
    )
    parser.add_argument(
        '--port',
        action='store',
        type=int,
        dest='port',
        default=27217,
        help='The port mongodb is runnung on'
    )
    parser.add_argument(
        '-t', '--connect-timeout-ms',
        dest='connectTimeoutMS', type=int,
        help='Timeout for connection to mongo[ds] instance (in milliseconds)',
        default=500,
        action='store'
    )
    parser.add_argument(
        '--replica-set',
        action='store',
        type=str,
        dest='replicaSet',
        help='The name of replicaset (mandatory for replset_state action)'
    )
    parser.add_argument(
        '-A', '--action',
        action='store',
        dest='action',
        default='connect',
        help='The action you want to take',
        choices=[
            "connect",
            "shard_balancer_state",
            "mongos_statistics"
         ]
    )
    parser.add_argument(
        '--secrets',
        dest='secrets',
        action='store',
        type=str,
        default='all',
        help='Section in secrets file with username and password'
    )
    parser.add_argument(
        '--warn',
        dest='warn_lag',
        action='store',
        type=int,
        default=70,
        help='Warning replication lag walue'
    )
    parser.add_argument(
        '--crit',
        dest='crit_lag',
        action='store',
        type=int,
        default=120,
        help='Critical replication lag walue'
    )
    return parser.parse_args()


def read_secrets(section):
    #
    # Reads username and password from specified section of secrets file.
    # If there is no file, or file is empty - sets username and password to ''
    #
    try:
        config = ConfigParser.ConfigParser({'username': '', 'password': ''})
        config.read(["/etc/mongodb.d/db-configs/secrets"])
        username = config.get(section, "username")
        password = config.get(section, "password")
    except ConfigParser.NoSectionError:
        username = 'monitor'
        password = 'monitor'
    return username, password


def solomon_metrics_from_dict_generator(d, prefix=''):
    """
    Recursively yield solomon format dict from nested dict
    """
    for key, value in d.iteritems():
        key.replace(" ", "_").replace("(", "_").replace(")", "_")
        if isinstance(value, dict):
            for data in solomon_metrics_from_dict_generator(value, prefix=(prefix + key + '.')):
                yield data
        else:
            try:
                yield {"sensor": prefix + key, "value": int(value), "project": "mongo", "service": "mongos"}
            except (ValueError, TypeError) as e:
                log.debug("Can't cast %s%s:%s to integer: %s" % (
                    prefix, key, value, e))


def solomon_metrics_from_dict_as_list(d, prefix=''):
    return list(solomon_metrics_from_dict_generator(d, prefix))


def graphite_metrics_from_dict(d, prefix=''):
    """
    Recursively convert dict to list of strings to send to graphite
    """
    metrics = []
    for key, value in d.iteritems():
        key = key.replace(" ", "_").replace("(", "_").replace(")", "_")
        if isinstance(value, dict):
            metrics += graphite_metrics_from_dict(
                value,
                prefix=(prefix + key + '.'))
        else:
            try:
                metric = ['%s %s %s\n' % (
                    prefix + key,
                    int(value),
                    int(time.time()))]
                log.debug(metric)
                metrics += metric
            except (ValueError, TypeError) as e:
                log.debug("Can't cast %s%s:%s to integer: %s" % (
                    prefix, key, value, e))
    return metrics


def send_metrics_to_graphite_client(metrics):
    """
    Send metrics to local graphite queue
    """
    status = False

    # Open sock from server
    sock = None
    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.settimeout(1.0)
        sock.connect(('localhost', 42001))
        log.debug('Send %d params to sender' % len(metrics))
        send_count = 0
        for metric in metrics:
            sock.send(metric)
            send_count += 1
        status = True
        log.debug('Sucess send %d metric to sender' % send_count)
    except Exception:
        status = False
    finally:
        if sock:
            sock.close()
    return status


def check_mongo_connection(host='127.0.0.1', port=27217, connectTimeoutMS=500):
    try:
        start_time = time.time()
        client = pymongo.MongoClient(
            host=host, port=port,
            connectTimeoutMS=connectTimeoutMS
        )
        client.admin.command("ismaster")
        stop_time = time.time()
        log.debug("Connection took {} seconds.".format(
            str(stop_time - start_time)))
        return (0, 'OK')
    except pymongo.errors.ConnectionFailure as e:
        return (2, "{}".format(e))


def check_mongo_shard_balancer_state(
    username='monitor',
    password='monitor',
    host='127.0.0.1',
    port=27217,
    connectTimeoutMS=500
):
    try:
        # Connect to mongod instance (not as replset client, because it will
        # send all command to PRIMARY of replicaset)
        client = pymongo.MongoClient(
            host=host, port=port,
            connectTimeoutMS=connectTimeoutMS
        )
        client.admin.authenticate(username, password)
        balancer_state = False
        settings_balancer = client.config.settings.find_one(
            {"_id": "balancer"})
        if settings_balancer is None:
            balancer_state = True
        else:
            if settings_balancer['stopped'] is True:
                balancer_state = False
            else:
                balancer_state = True
        if balancer_state:
            return (0, "Balancer is running")
        else:
            return (2, "Balancer is stopped")
    except pymongo.errors.ConnectionFailure as e:
        return (2, "ConnectionFailure: {}".format(e))


def get_databases_stats(client, port):
    '''
    Iterate over all dbs and get stats from them.
    '''
    database_stats_list = []
    for database in client.database_names():
        db = client[database]
        database_stats = db.command('dbstats')

        # Delete per shard info, we not interested in that.
        del database_stats[u'raw']

        log.debug(database, database_stats)
        database_stats_list.append((database, database_stats))
    return database_stats_list


def check_mongos_statistics(
    username="monitor",
    password="monitor",
    host="127.0.0.1",
    port=27217,
    connectTimeoutMS=500  # noqa
):
    try:
        # Connect to mongod instance (not as replset client, because it will
        # send all command to PRIMARY of replicaset)
        client = pymongo.MongoClient(
            host=host, port=port,
            connectTimeoutMS=connectTimeoutMS
        )
        client.admin.authenticate(username, password)
        # Use the same connection to get different metrics.
        server_status = client.admin.command({"serverStatus": 1})
        log.debug(server_status)
        server_metrics = graphite_metrics_from_dict(
            server_status,
            prefix="one_min.{}.mongo.{}.".format(socket.getfqdn().replace(
                ".", "_"), port)
        )

        database_stats_list = get_databases_stats(client, port)
        conn_pool_stats = get_conn_pool_stats(client, port)
        shard_conn_pool_stats = get_shard_conn_pool_stats(client, port)

        metrics = []
        metrics += server_metrics
        metrics += graphite_metrics_from_dict(
            conn_pool_stats,
            prefix="one_min.{}.mongo.{}.conn_pool_stats.".format(
                socket.getfqdn().replace(".", "_"), port)
        )
        metrics += graphite_metrics_from_dict(
            shard_conn_pool_stats,
            prefix="one_min.{}.mongo.{}.shard_conn_pool_stats.".format(
                socket.getfqdn().replace(".", "_"), port)
        )

        solomon_sensors = []

        for database, database_stats in database_stats_list:
            metrics += graphite_metrics_from_dict(
                database_stats,
                prefix='one_min.{}.mongo.{}.dbstats.{}.'.format(
                    socket.getfqdn().replace('.', '_'), port, database)
            )
            solomon_sensors += solomon_metrics_from_dict_as_list(database_stats)

        solomon_sensors += solomon_metrics_from_dict_as_list(conn_pool_stats)
        solomon_sensors += solomon_metrics_from_dict_as_list(shard_conn_pool_stats)

        solomon.send_sensors(solomon_sensors)
        return 0, "Successfully sent data to solomon"
    except pymongo.errors.ConnectionFailure as e:
        return 2, "ConnectionFailure: {}".format(e)


def get_conn_pool_stats(client, port):
    """Get connection pool stats.

    The command connPoolStats returns information
    regarding the open outgoing connections
    from the current database instance to other members of the cluster or replica set.
    https://docs.mongodb.com/v3.2/reference/command/connPoolStats/

    :param client: Mongo client
    :type client: pymongo.MongoClient
    :param port: Mongo instance port
    :type port: int

    :return: Graphite-formatted metrics
    :rtype: list
    """

    try:
        return client.admin.command({"connPoolStats": 1})
    except Exception as err:
        log.info("{};{}".format(2, "CmdExecError: {}".format(err)))
    return []


def get_shard_conn_pool_stats(client, port):
    """Get shard connection pool stats.

    Information on the pooled and cached connections in the sharded connection pool.
    The command also returns information on the per-thread connection cache
    in the connection pool.
    https://docs.mongodb.com/v3.2/reference/command/shardConnPoolStats/

    :param client: Mongo client
    :type client: pymongo.MongoClient
    :param port: Mongo instance port
    :type port: int

    :return: Graphite-formatted metrics
    :rtype: list
    """

    try:
        return client.admin.command({"shardConnPoolStats": 1})
    except Exception as err:
        log.info("{};{}".format(2, "CmdExecError: {}".format(err)))
    return []


def to_monrun(code, message):
    log.info("{};{}".format(code, message))


def main():
    args = parse_args()
    if args.verbose:
        log.setLevel(logging.DEBUG)

    (username, password) = read_secrets(args.secrets)
    if args.action == 'connect':
        (code, message) = check_mongo_connection(
            host=args.host, port=args.port,
            connectTimeoutMS=args.connectTimeoutMS
        )
        to_monrun(code, message)
    elif args.action == "shard_balancer_state":
        (code, message) = check_mongo_shard_balancer_state(
            username=username,
            password=password,
            host=args.host, port=args.port,
            connectTimeoutMS=args.connectTimeoutMS
        )
        to_monrun(code, message)
    elif args.action == "mongos_statistics":
        (code, message) = check_mongos_statistics(
            username=username,
            password=password,
            host=args.host, port=args.port,
            connectTimeoutMS=args.connectTimeoutMS
        )
        to_monrun(code, message)


if __name__ == "__main__":
    #
    # Main App
    #
    main()

