import os
import time
import socket
import getpass
import logging
import argparse
import requests
from collections import OrderedDict

import nanny_rpc_client
import metrika.pylib.log

from infra.awacs.proto import model_pb2, api_stub, api_pb2


AWACS_URL = os.getenv('AWACS_URL', 'https://awacs.yandex-team.ru/api/')


def get_cgroup_info(cgroup):
    groups2hosts = requests.get("https://c.yandex-team.ru/api/groups2hosts/{}".format(cgroup)).text.rstrip().split("\n")
    data = OrderedDict()
    print groups2hosts
    for host in groups2hosts:
        hosts_info = requests.get("https://c.yandex-team.ru/api/hosts/{}".format(host)).text
        if '<root_datacenter>sas</root_datacenter>' in hosts_info:
            dc = 'sas'
        elif '<root_datacenter>vla</root_datacenter>' in hosts_info:
            dc = 'vla'
        elif '<root_datacenter>man</root_datacenter>' in hosts_info:
            dc = 'man'
        elif '<root_datacenter>myt</root_datacenter>' in hosts_info:
            dc = 'myt'
        elif '<root_datacenter>iva</root_datacenter>' in hosts_info:
            dc = 'iva'
        else:
            dc = 'none'

        if dc == 'none':
            for letter, dc_candidate in [('w', 'man'), ('f', 'myt'), ('h', 'sas'), ('v', 'vla'), ('e', 'iva')]:
                if host.endswith('{}.paysys.yandex.net'.format(letter)) or \
                   host.endswith('{}.yandex.ru'.format(letter)):
                    print "Rewritten DC for {} from none to {}".format(host, dc_candidate)
                    dc = dc_candidate

        data.setdefault(dc, [])
        data[dc].append(host)

    return data


def get_ip6_by_fqdn(fqdn):
    print "get_ip6_by_fqdn", fqdn
    ips = set(
        [x[4][0] for x in socket.getaddrinfo(fqdn, None, socket.AF_INET6, socket.SOCK_STREAM)]
    )
    if len(ips) != 1:
        print "For ip {} resolved not one address: {}".format(fqdn, ips)
        print "Exiting!"
        exit(1)
    else:
        return ips.pop()


class AwacsBackend(object):
    def __init__(self, namespace, backend_id, hosts, port, user):
        self.namespace = namespace
        self.backend_id = backend_id
        self.hosts = hosts
        self.port = port
        self.user = user
        self.logger = logging.getLogger(self.__class__.__name__)
        self.token = os.environ.get('AWACS_OAUTH_TOKEN')
        self.nanny_client = nanny_rpc_client.RequestsRpcClient(AWACS_URL,
                                                               request_timeout=10,
                                                               oauth_token=self.token)
        self.backends_stub = api_stub.BackendServiceStub(self.nanny_client)
        self.endpoint_set_stub = api_stub.EndpointSetServiceStub(self.nanny_client)

    def populate_endpoint_set(self, endpointset_pb, hosts):
        for fqdn in hosts:
            ip = get_ip6_by_fqdn(fqdn)
            endpointset_pb.spec.instances.add(
                host=fqdn,
                port=self.port,
                weight=1,
                ipv6_addr=ip
            )
        return endpointset_pb

    def create_backend(self):
        self.logger.error("Backend {} not found. Going to create it.".format(self.backend_id))
        selector = model_pb2.BackendSelector(
            type=model_pb2.BackendSelector.MANUAL,
        )
        spec_pb = model_pb2.BackendSpec(selector=selector)
        req_pb = api_pb2.CreateBackendRequest(spec=spec_pb)
        req_pb.meta.id = self.backend_id
        req_pb.meta.namespace_id = self.namespace
        req_pb.meta.comment = 'Created from awacs_backend_updater'
        req_pb.meta.auth.type = req_pb.meta.auth.STAFF
        result = self.backends_stub.create_backend(req_pb)
        self.logger.info("Backend `{}` created: {}".format(self.backend_id, result))

    def create_endpoint_set(self, backend_pb):
        req_pb = api_pb2.CreateEndpointSetRequest()

        # указываем версию бэкенда, для которой будет актуальна новая ревизия эндпоинтсета
        req_pb.backend_version = backend_pb.meta.version
        req_pb.meta.id = self.backend_id
        req_pb.meta.namespace_id = self.namespace
        req_pb.meta.auth.type = req_pb.meta.auth.STAFF
        req_pb = self.populate_endpoint_set(req_pb, self.hosts)
        resp_pb = self.endpoint_set_stub.create_endpoint_set(req_pb)
        return resp_pb

    def update_endpoint_set(self, backend_pb, endpoint_set_pb):
        # если эндпоинтсет существует, обновляем список инстансов
        req_pb = api_pb2.UpdateEndpointSetRequest()
        # указываем версию бэкенда, для которой будет актуальна новая ревизия эндпоинтсета
        req_pb.backend_version = backend_pb.meta.version
        req_pb.meta.id = self.backend_id
        req_pb.meta.namespace_id = self.namespace
        # указываем версию эндпоинтсета, которую хотим обновить
        req_pb.meta.version = endpoint_set_pb.meta.version
        req_pb = self.populate_endpoint_set(req_pb, self.hosts)
        resp_pb = self.endpoint_set_stub.update_endpoint_set(req_pb)
        return resp_pb

    def get_endpoint_set(self):
        req_pb = api_pb2.GetEndpointSetRequest(
            id=self.backend_id,
            namespace_id=self.namespace
        )
        resp_pb = self.endpoint_set_stub.get_endpoint_set(req_pb)
        endpoint_set_pb = resp_pb.endpoint_set
        return endpoint_set_pb

    def get_backend(self):
        get_req_pb = api_pb2.GetBackendRequest(
            id=self.backend_id,
            namespace_id=self.namespace
        )
        resp_pb = self.backends_stub.get_backend(get_req_pb)
        return resp_pb

    def update(self):
        try:
            # получаем бэкенд
            resp_pb = self.get_backend()
            self.logger.info("Backend {} exists".format(self.backend_id))
        except nanny_rpc_client.exceptions.NotFoundError:
            # если backend не существует, создаем его
            self.logger.error("Backend {} not found. Going to create it.".format(self.backend_id))
            self.create_backend()
            time.sleep(2)  # In case not fast api
            # resp_pb = self.backends_stub.get_backend(get_req_pb)
            resp_pb = self.get_backend()
            self.logger.info("Created backend:\n{}".format(resp_pb.backend))
        backend_pb = resp_pb.backend

        if not backend_pb.spec.selector.type == backend_pb.spec.selector.MANUAL:
            self.logger.error("You can modify only MANUAL type of backends. `{}` is not MANUAL".format(self.backend_id))
            exit(1)

        try:
            # пробуем получить его эндпоинтсет (идентификатор которого совпадает с идентификатором бэкенда)
            endpoint_set_pb = self.get_endpoint_set()
            current_instances = endpoint_set_pb.spec.instances
            self.logger.info("EndpointSet {} exits".format(self.backend_id))
            self.logger.info("Got current instances:\n{}".format(current_instances))
        except nanny_rpc_client.exceptions.NotFoundError:
            self.logger.info("EndpointSet {} not found. Going to create it".format(self.backend_id))
            resp_pb = self.create_endpoint_set(backend_pb)
        else:
            self.logger.info("Going to update endpoint_set")
            resp_pb = self.update_endpoint_set(backend_pb, endpoint_set_pb)
        self.logger.info("Got endpoint_set spec in responce:\n{}".format(resp_pb.endpoint_set.spec))


def main():
    metrika.pylib.log.init_logger('', stdout=True)

    parser = argparse.ArgumentParser()
    parser.add_argument('-g', '--cgroup', required=True, help="Conductor group name")
    parser.add_argument('-n', '--namespace', required=True, help="AWACS namespace_id")
    parser.add_argument('-b', '--backend', required=True, help="Backend name")
    parser.add_argument('-p', '--port', required=True, type=int, help="Backend port")
    parser.add_argument('-u', '--user', help="User to get from ssh-agent, default getpass.getuser()", default=getpass.getuser())
    args = parser.parse_args()

    logger = logging.getLogger('main')
    logger.info("Got next params:\n\tAWACS namespace id: {}\n\tConductor group: {}\n\tBackend port: {}".format(args.namespace, args.cgroup, args.port))

    conductor_info = get_cgroup_info(args.cgroup)
    logger.info("Got conductor info: {}".format(conductor_info))
    for dc_name in conductor_info:
        backend_id = "{backend_name}-{dc_name}".format(backend_name=args.backend, dc_name=dc_name)
        backend_hosts = [host for host in conductor_info[dc_name]]
        awacs_backend = AwacsBackend(args.namespace, backend_id, backend_hosts, args.port, args.user)
        awacs_backend.update()


if __name__ == '__main__':
    main()
