# coding=utf-8
# This script generates a set of shell commands to create databases
# for https://st.yandex-team.ru/PAYSYSADMIN-7815
#
# First you need to create said databases
# $ python yc_create_database.sh.py db > db.sh
# $ bash db.sh
#
# After successful creation you add extensions to them
# $ python yc_create_database.sh.py extensions > extensions.sh
# $ bash extensions.sh
#
# Before creating users make request to MDB to add database to SOX
# https://st.yandex-team.ru/MDBSUPPORT-2932
# $ python yc_create_database.sh.py users > users.sh
# $ bash users.sh
#
# After that - launch finishing sequence of sql commands
# https://st.yandex-team.ru/PAYSYSADMIN-7815#5fa00bbdd462e0191d3c8351

from collections import OrderedDict
import sys
import random
import string

TICKET = "PAYSYSADMIN-7815"

PG_VERSION = "12"
PG_EXTENSIONS = "pg_repack=1,pg_stat_statements=1,pg_stat_kcache=1"
CONTOURS_TO_ENV = OrderedDict((
    ("prod", ("stable", "robot-ps-prod-salt")),
    ("test", ("testing", "robot-ps-test-salt")),
    ("load", ("load", "robot-ps-test-salt")),
    ("dev", ("unstable", "robot-ps-test-salt"))
))

ZONES = "--host zone-id=man --host zone-id=vla --host zone-id=sas"

# contour -> (preset, yc env)
DEFAULT_PRESETS = OrderedDict((
    ("test", ("s3.nano", "production")),
    ("load", ("s3.nano", "production")),
    ("dev", ("s3.nano", "production")),
    ("prod", ("s3.small", "production")),
))

USER_CONNECTION_LIMITS_SMALL = OrderedDict((
    ("prod", "5"),
    ("test", "5"),
    ("load", "5"),
    ("dev", "5")
))

USER_CONNECTION_LIMITS_BIG = OrderedDict((
    ("prod", "500"),
    ("test", "100"),
    ("load", "100"),
    ("dev", "100")
))

DB_TYPES = [
    {
        "type": "gomer",
        "desc": "PAYSYSADMIN-7815 База данных Гомера",
        "preset": DEFAULT_PRESETS,
        "sizes": OrderedDict((("test", "40"), ("load", "40"), ("dev", "40"), ("prod", "40"),)),
        "schemas": ["tpay", "tuti"],
        "auto_users": [
            ("trust_gomer", "tpay,public", USER_CONNECTION_LIMITS_SMALL, False),
            ("trust_gomer_app", "tpay,public", USER_CONNECTION_LIMITS_BIG, True),
            ("trust_utility", "tuti", USER_CONNECTION_LIMITS_SMALL, True)
        ],
        "idm_users": OrderedDict((("prod", ["gvedeneev", "sunshineguy"]),)),
    },
    {
        "type": "metadata",
        "desc": "PAYSYSADMIN-7815 База данных Партнеров и продуктов",
        "preset": DEFAULT_PRESETS,
        "sizes": OrderedDict((("test", "10"), ("load", "10"), ("dev", "10"), ("prod", "10"))),
        "schemas": ["tpay", "tuti"],
        "auto_users": [
            ("trust_metadata", "tpay,public", USER_CONNECTION_LIMITS_SMALL, False),
            ("trust_metadata_app", "tpay,public", USER_CONNECTION_LIMITS_BIG, True),
            ("trust_utility", "tuti", USER_CONNECTION_LIMITS_SMALL, True)
        ],
        "idm_users": OrderedDict((("prod", ["sunshineguy"]),)),
    },
]


def create_user_password_command(db_type, username, salt_env, robot_name, password):
    salt_filename = "postgres_{db_type}_{username}-shared1.txt".format(
        db_type=db_type, username=username
    )
    cmd = "SECRET_ID=$(ya vault create secret {salt_filename} " + \
          "-t secret:auto,role:trust,environment:{salt_env} " + \
          "-r +reader:user:{robot_name} " + \
          "+owner:abc:paysys-admin:scope:administration " + \
          "-k default_value={password} " + \
          "-c '{db_type} {contour}/{salt_env} {username} password {ticket}' -j | jq .uuid -r)"

    return cmd.format(
        salt_filename=salt_filename, salt_env=salt_env,
        robot_name=robot_name, password=password,
        contour=contour, db_type=db_type, username=username,
        ticket=TICKET
    )


def generate_password():
    letters = string.ascii_lowercase + string.ascii_uppercase + string.digits
    password = ''.join(random.choice(letters) for i in range(32))
    return password


if sys.argv[1] == "help":
    print "{} <stage>".format(sys.argv[0])
    print "db - create database script"
    print "extensions - set extensions script"
    print "users - create users script"
    print "- other methods (db_create, db_delete, users_update) created to implement database renaming"

# yc managed-postgresql cluster create --name trust_gomer_dev
# --description "PAYSYSADMIN-7815 База данных Гомера"
# --environment prestable
# --network-id ' '
# --host zone-id=man --host zone-id=vla --host zone-id=sas
# --database name=trust_gomer,owner=trust_gomer
# --disk-type local-ssd --disk-size 40
# --user name=trust_gomer,password=
# --postgresql-version 12 --resource-preset s3.nano
for k in DB_TYPES:
    db_type = k.get("type")
    db_owner = "trust_{}".format(db_type)
    for contour, preset_settings in k.get("preset").iteritems():
        preset, environment = preset_settings
        db_name = "trust_{}_{}".format(db_type, contour)
        db_name_old = "trust_{}".format(db_type)
        cluster_name = "trust_{}_{}".format(db_type, contour)
        cluster_description = k.get("desc")
        cluster_size = k.get("sizes").get(contour)
        salt_env, robot_name = CONTOURS_TO_ENV.get(contour)

        db_owner_password = ""

        if sys.argv[1] == "db":
            print "set -ex"
            print create_user_password_command(
                db_type, db_owner, salt_env, robot_name,
                password=generate_password()
            )

            cmd = "yc managed-postgresql cluster create --name {cluster_name} " + \
                  "--description '{cluster_description}' " + \
                  "--network-id ' ' " + ZONES + " " + \
                  "--database name={db_name},owner={db_owner} " + \
                  "--environment {environment}" + \
                  "--disk-type local-ssd --disk-size {cluster_size} " + \
                  "--user name={db_owner},password=$(ya vault get version $SECRET_ID -o default_value) " + \
                  "--postgresql-version {pg_version} --resource-preset {preset}"
            print cmd.format(
                cluster_name=cluster_name, cluster_description=cluster_description,
                db_name=db_name, db_owner=db_owner, cluster_size=cluster_size,
                pg_version=PG_VERSION, preset=preset, environment=environment
            )

            print
        if sys.argv[1] == "db_create":
            print "set -ex"
            cmd = "yc managed-postgresql database create {db_name} " + \
                  "--owner {db_owner} " + \
                  "--cluster-name {cluster_name} " + \
                  "--extensions {extensions}"
            print cmd.format(
                db_name=db_name, db_owner=db_owner,
                cluster_name=cluster_name,
                extensions=PG_EXTENSIONS
            )

        if sys.argv[1] == "db_delete":
            print "set -ex"
            cmd = "yc managed-postgresql database delete {db_name_old} " + \
                  "--cluster-name {cluster_name} "
            print cmd.format(
                db_name_old=db_name_old,
                cluster_name=cluster_name
            )

        if sys.argv[1] == "extensions":
            print "set -ex"
            cmd = "yc managed-postgresql database update {db_name} " + \
                  "--cluster-name {cluster_name} " + \
                  "--extensions {extensions}"
            print cmd.format(
                db_name=db_name, cluster_name=cluster_name,
                extensions=PG_EXTENSIONS
            )

        if sys.argv[1] == "users":
            print "set -ex"
            for user_name, schemas, conn_limit_settings, create_on_users_stage in k.get("auto_users"):
                conn_limit = conn_limit_settings.get(contour)
                if create_on_users_stage:
                    print create_user_password_command(
                        db_type, user_name, salt_env, robot_name,
                        password=generate_password()
                    )
                    cmd = "yc managed-postgresql user create {user_name} " + \
                          "--cluster-name {cluster_name} " + \
                          "--password=$(ya vault get version $SECRET_ID -o default_value) " + \
                          "--permissions={db_name} "

                    print cmd.format(
                        user_name=user_name, cluster_name=cluster_name,
                        db_name=db_name, conn_limit=conn_limit
                    )

                cmd = "yc managed-postgresql user update {user_name} " + \
                      "--cluster-name {cluster_name} " + \
                      "--grants=mdb_admin,reader "+ \
                      "--conn-limit={conn_limit}"
                print cmd.format(
                    user_name=user_name, cluster_name=cluster_name,
                    conn_limit=conn_limit
                )
                print

            for user_name in k.get("idm_users", {}).get(contour, []):
                cmd = "yc managed-postgresql user update {user_name} " + \
                      "--cluster-name {cluster_name} " + \
                      "--grants=mdb_admin,reader "
                print cmd.format(
                    user_name=user_name, cluster_name=cluster_name
                )
                print

        if sys.argv[1] == "users_update":
            print "set -ex"
            for user_name, schemas, conn_limit_settings, create_on_users_stage in k.get("auto_users"):
                conn_limit = conn_limit_settings.get(contour)
                cmd = "yc managed-postgresql user update {user_name} " + \
                      "--cluster-name {cluster_name} " + \
                      "--permissions {db_name} " + \
                      "--grants=mdb_admin,reader " + \
                      "--conn-limit={conn_limit}"
                print cmd.format(
                    user_name=user_name, db_name=db_name,
                    cluster_name=cluster_name,
                    conn_limit=conn_limit
                )
                print

            for user_name in k.get("idm_users", {}).get(contour, []):
                cmd = "yc managed-postgresql user update {user_name} " + \
                      "--cluster-name {cluster_name} " + \
                      "--permissions {db_name} " + \
                      "--grants=mdb_admin,reader "
                print cmd.format(
                    user_name=user_name,
                    cluster_name=cluster_name,
                    db_name=db_name
                )
                print
