import argparse
import hashlib
import logging
import os
import sys

import ipaddr
import requests
import rtapi
import urllib3

log = logging.getLogger('gen_ip')


def parse_args():
    parser = argparse.ArgumentParser(
        description='Generate ip address for new host'
    )
    parser.add_argument(
        '--fqdn', required=True, help='FQDN of new host'
    )

    parser.add_argument(
        '--inventory_id', help='Eine host inventory id'
    )

    global_group = parser.add_mutually_exclusive_group()
    global_group.add_argument(
        '--project_id', help='Project id of new host'
    )
    global_group.add_argument(

        '--vlan', help='VLAN from racktables'
    )

    project_id_arg_group = parser.add_argument_group(
        'project_id', 'Project id specific options'
    )
    project_id_arg_group.add_argument(
        '--project_id_location_prefix',
        help='Specify location ipv6 prefix explicitly via this option' +
             '(for example 2a02:6b8:c03:717::/64)'
    )
    project_id_arg_group.add_argument(
        '--project_id_iface', default='br0',
        help='Project ID RA interface',
    )

    vlan_arg_group = parser.add_argument_group('vlan', 'VLAN specific options')
    vlan_arg_group.add_argument(
        '--datacenter', help='Datacenter to generate ip for'
    )
    parser.add_argument(
        '--debug', action='store_true', help='enable debug logging'
    )
    return parser.parse_args()


def print_availability_notice(ip):
    """Print notice about neccessarity of checking ip before using it"""
    print(
        '''
        NOTICE: Please don't forget to check that ip is really avalivable. Use
        the following commands:

        host {0}
        ping6 {0}
        '''.format(ip)
    )


def test_fqdn(fqdn):
    """
    Test if zone for the new record is one of the allowed.
    Show warning for conditionaly allowed zones (for example old zones that
    can't be used for the new host creation)
    """
    allowed_zones = ['.paysys.yandex.net']
    warning_zones = [
        '.yandex.ru',
        '.pci.yandex.net',
    ]

    for zone in allowed_zones:
        if fqdn.endswith(zone):
            return True

    for zone in warning_zones:
        if fqdn.endswith(zone):
            print(
                'WARNING: use zone {} only if you are really know what are you doing'.format(zone)
            )
            return True

    return False


def craft_project_id_ip(network, project_id, host_id):
    """
    Generate project id ip for given network, project id and host id.

    Use mechanics of addition integet to ipaddr.IPv6Address object
    """
    ip = network.ip + int(host_id, 16) + int(project_id, 16) * 0x10 ** 8
    return ip


def get_network_list_per_dc_from_racktables():
    response = urllib3.urlopen(
        'https://racktables.yandex-team.ru/export/networklist.php?report=perdc'
    )

    return response.readlines()


def main():
    args = parse_args()
    if args.debug:
        logging.basicConfig(level=logging.DEBUG)

    if not test_fqdn(args.fqdn):
        print('Hostname cannot be used')
        sys.exit(1)

    host_id = hashlib.md5(args.fqdn.encode('utf-8')).hexdigest()[0:8]

    log.debug('Got host id {}'.format(host_id))

    if args.project_id is not None:
        if args.project_id_location_prefix is not None:
            prefix = args.project_id_location_prefix
        else:
            host_switch = get_host_switch_from_eine(args.fqdn)
            if host_switch is None and args.inventory_id:
                host_switch = get_host_switch_from_eine(inventory=args.inventory_id)
            if host_switch:
                prefix = get_switch_vlan333_network(host_switch)
            else:
                log.error('Make sure fqdn or inventory_id is present in Eine. Or specify --project_id_location_prefix')
                exit(1)

        log.debug('IPv6 prefix is {}'.format(prefix))

        network = ipaddr.IPv6Network(prefix)

        ip = craft_project_id_ip(network, args.project_id, host_id)

        print('IP is {}'.format(ip))
        print_availability_notice(ip)

    elif args.vlan is not None:
        for r_line in get_network_list_per_dc_from_racktables():
            rs_line = r_line.split()
            if rs_line[1] == str(args.vlan):
                prefix = rs_line[0]
                try:
                    network = ipaddr.IPv6Network(prefix)
                except ipaddr.AddressValueError:
                    continue
                log.debug('IPv6 prefix is {}'.format(prefix))
                ip = network.ip + int(host_id, 16)
                print('IP for DC {} is {}'.format(rs_line[2], ip))
                print_availability_notice(ip)

        print(
            '''
            WARNING: there are more than one address because it is hard
            to filter DC by it's full russian hame. Please choose one
            with DC identifier you need.
            '''
        )


def get_switch_vlan333_network(switch):
    api = rtapi.RTAPI()
    object_id = api.call.findObjectByName(switch)
    switch_info = api.call.getObjectIPAllocationList(object_id)
    for ipbin, args in switch_info.items():
        if args.get('osif') == 'vlanif333':
            network = api.call.spotNetworkByIP(ipbin)
            return f'{network.get("ip_bin")}/{network.get("mask")}'


def get_host_switch_from_eine(host: str = None, inventory: int = None):
    token = os.environ.get('EINE_TOKEN')
    if token is None:
        log.error('EINE_TOKEN should be specified')
        exit(1)
    headers = {
        'Authorization': f'OAuth {token}'
    }
    if host is not None:
        url = f'https://eine.yandex-team.ru/api/v2/computers/hostname/{host}/props/switch'
    elif inventory is not None:
        url = f'https://eine.yandex-team.ru/api/v2/computers/inventory/{inventory}/props/switch'
    else:
        raise AttributeError('Either host or inventory should be specified')

    result = requests.get(url, headers=headers).json()
    switch_port = result.get('result', {}).get('value')
    if switch_port is None:
        return
    return switch_port.split()[0]
