import sys
import argparse
import importlib

import logging
import logging.handlers
import urllib2
import httplib
import coloredlogs

from collections import namedtuple
from juggler_sdk.check_sync import FullCheckDifference

import paysys.sre.tools.monitorings.configs as configs

from paysys.sre.tools.monitorings.lib.util import yasm_template
from paysys.sre.tools.monitorings.lib.util.helpers import gen_checks_for_host, get_juggler_api, get_solomon_token
from paysys.sre.tools.monitorings.lib.util.yasm import YasmAlertApiExtended, yasm_gen_alerts_for_host
from paysys.sre.tools.monitorings.lib.util.solomon import SolomonAlertingService, solomon_gen_alerts_for_host

ConfigInfo = namedtuple('ConfigInfo', ['host', 'config', 'env'])


class MonitoringsApplier:
    def __init__(self, options):
        self.options = options
        _projects_args = options.project
        if not isinstance(options.project, list):
            _projects_args = [options.project]
        self.projects = _projects_args

        _environments_args = options.environment
        if not isinstance(options.environment, list):
            _environments_args = [options.environment]
        self.environments = _environments_args

        _config_args = options.config
        if not isinstance(options.config, list):
            _config_args = [options.config]
        self.configs = _config_args

    @staticmethod
    def _get_check_attrs_from_module_name(module_name):
        _, module, project, environment, host = module_name.rsplit(".", 4)

        return module, project, environment, host

    @staticmethod
    def _get_path_from_host(module_name):
        _, module, project, environment, host = module_name.rsplit(".", 4)
        return "{}.{}.{}.{}".format(module, project, environment, host)

    @staticmethod
    def _get_path_from_environment(module_name):
        _, module, project, environment = module_name.rsplit(".", 3)
        return "{}.{}.{}".format(module, project, environment)

    @staticmethod
    def _get_path_from_project(module_name):
        _, module, project = module_name.rsplit(".", 2)
        return "{}.{}".format(module, project)

    def _check_module(self, config):
        if len(config.__name__.split(".")) < 4:
            return False

        module, project, environment, host = self._get_check_attrs_from_module_name(config.__name__)
        check_path = self._get_path_from_host(config.__name__)

        if module != "configs":
            return False

        if environment == "base":
            logging.info("Config [{0}] looks like base environment, skipped".format(check_path))
            return False

        keys = config.__dict__.keys()
        if 'checks' in keys and 'host' in keys and 'children' in keys:
            return True
        else:
            logging.error("Cannot parse config file [{0}], skipped".format(check_path))
            return False

    @staticmethod
    def _log_juggler_check_result(check, result):
        if result.changed:
            if isinstance(result.diff, FullCheckDifference):
                logging.info('juggler: {0}:{1} - [added]'.format(check.host, check.service))
            else:
                logging.info('juggler: {0}:{1} - [updated]:'.format(check.host, check.service))
                for field in result.diff._fields:
                    logging.info(
                        "juggler: Changed [{0}]:\n"
                        "\t\told: {1}\n"
                        "\t\tnew: {2}".format(field.name, field.current, field.desired)
                    )
        else:
            logging.debug('juggler: {0}:{1} - [not changed]'.format(check.host, check.service))

    def process_projects(self):
        _projects = map(lambda x: importlib.import_module(configs.__name__ + "." + x), configs.PROJECTS)

        if self.projects != ['all']:
            _projects = [
                importlib.import_module(configs.__name__ + "." + project)
                for project in self.projects
                if project in configs.PROJECTS
            ]

        logging.debug("Found projects: {}".format(map(lambda x: x.__name__, _projects)))

        logging.info(
            "Valid projects for apply: [{0}]".format(
                ", ".join(
                    [self._get_path_from_project(project.__name__) for project in _projects]
                )
            )
        )

        for project in _projects:
            logging.info(
                "Applying configuration for project [{0}]".format(
                    self._get_path_from_project(project.__name__)
                )
            )
            solomon_alerts_service = SolomonAlertingService(
                token=get_solomon_token(),
                is_dry_run=self.options.dry,
                no_cleanup=self.options.no_cleanup,
            )

            self._process_environments(project, solomon_alerts_service)

            solomon_alerts_service.operate()

        logging.info("Done processing projects")

    def _process_environments(
        self,
        project,
        solomon_alerts_service,
    ):
        _environments = map(lambda x: importlib.import_module(project.__name__ + "." + x), project.ENVIRONMENTS)

        if self.environments != ['all']:
            _environments = [
                importlib.import_module(project.__name__ + "." + environment)
                for environment in self.environments
                if environment in project.ENVIRONMENTS
            ]

        logging.debug("Found environments: {}".format(map(lambda x: x.__name__, _environments)))

        logging.info(
            "Valid environments for apply in project: [{0}]".format(
                ", ".join(
                    [self._get_path_from_environment(env.__name__) for env in _environments]
                )
            )
        )

        for environment in _environments:
            logging.info(
                "Applying configuration from [{0}]".format(
                    self._get_path_from_environment(environment.__name__)
                )
            )
            self._process_environment(
                environment,
                solomon_alerts_service,
            )

    def _process_environment(
        self,
        environment,
        solomon_alerts_service,
    ):
        _configs = map(lambda x: importlib.import_module(environment.__name__ + "." + x), environment.CONFIGS)

        if self.configs != ['all']:
            _configs = [
                importlib.import_module(environment.__name__ + "." + config)
                for config in self.configs
                if config in environment.CONFIGS
            ]

        logging.debug("Found configs: {}".format(map(lambda x: x.__name__, _configs)))

        for config in _configs:
            if self._check_module(config):
                self._process_config(
                    ConfigInfo(config.__name__.split(".")[-1], config, environment),
                    solomon_alerts_service,
                )

    def _parse_defaults(self, config_info):
        defaults = None
        if 'defaults' in config_info.config.__dict__.keys():
            defaults = config_info.config.defaults
            logging.debug(
                "Overrides default settings from [{0}/{1}]".format(
                    self._get_path_from_environment(config_info.env.__name__), config_info.host
                )
            )
        elif 'defaults' in config_info.env.__dict__.keys():
            defaults = config_info.env.defaults
            logging.debug(
                "Overrides default settings from [{0}]".format(
                    self._get_path_from_environment(config_info.env.__name__)
                )
            )
        else:
            logging.debug("Not found custom settings in env or host configs, use global defaults")
        return defaults

    def _apply_juggler_config(self, checks, config_info):
        juggler_api = get_juggler_api(
            "paysys_{0}_{1}".format(
                self._get_path_from_environment(config_info.env.__name__),
                config_info.host,
            ),
            self.options.dry
        )

        juggler_checks = gen_checks_for_host(
            config_info.config.host,
            config_info.config.children,
            checks,
            config_info.env.notifications,
            self._parse_defaults(config_info),
            config_info.config.split_by_dc if 'split_by_dc' in config_info.config.__dict__ else None
        )

        for juggler_check in sorted(juggler_checks, key=lambda x: x.service):
            result = juggler_api.upsert_check(juggler_check)
            self._log_juggler_check_result(juggler_check, result)

        if self.options.no_cleanup:
            logging.info('Juggler cleanup is skipped.')
        else:
            removed = juggler_api.cleanup()
            for check in removed.removed:
                logging.warn("juggler: {0}:{1} - [removed]".format(check[0], check[1]))

    def _apply_yasm_config(self, checks, config_info):
        if self.options.dry:
            logging.warn("Dry run not supported for yasm checks")
            return

        yasm_alerts = yasm_gen_alerts_for_host(config_info.config.host, checks)
        yasm_alerts_api = YasmAlertApiExtended(host=config_info.config.host, debug=False)

        for yasm_alert in sorted(yasm_alerts, key=lambda x: x.name):
            yasm_alerts_api.upsert_alert(yasm_alert)

        if self.options.no_cleanup:
            logging.info('YASM cleanup is skipped.')
        else:
            yasm_alerts_api.cleanup_alerts()

    def _apply_yasm_template_config(self, config_info):
        if hasattr(config_info.config, 'yasm_template'):
            alert_template = yasm_template.AlertTemplate.create_for_host(
                config_info.config.host, config_info.config.yasm_template()
            )
            yasm_template.upsert(config_info.config.host, alert_template, self.options.dry)
            # TODO(afrolovskiy): implement cleanup for alert templates

    def _process_config(self, config_info, solomon_alerts_service):
        logging.info(
            "Parsing config file [{0}/{1}]:".format(
                self._get_path_from_host(config_info.env.__name__),
                config_info.host
            )
        )

        checks = config_info.config.checks()
        self._apply_juggler_config(checks, config_info)
        self._apply_yasm_template_config(config_info)
        self._apply_yasm_config(checks, config_info)
        solomon_alerts_service.collect_alerts(
            solomon_gen_alerts_for_host(config_info.config.host, checks)
        )


def handle_exception(exc_type, exc_value, exc_traceback):
    if issubclass(exc_type, KeyboardInterrupt):
        sys.__excepthook__(exc_type, exc_value, exc_traceback)
        return
    logger = logging.getLogger()
    logger.error("Uncaught exception", exc_info=(exc_type, exc_value, exc_traceback))
    logger.fatal("Aborted")
    sys.exit(1)


def init_logger():
    logger = logging.getLogger()
    logger.propagate = False
    logger.setLevel(logging.INFO)
    coloredlogs.install(level=logging.INFO, logger=logger, fmt='%(asctime)s.%(msecs)03d %(levelname)s %(message)s')
    sys.excepthook = handle_exception

    return logger


def enable_debug():
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)
    coloredlogs.set_level(logging.DEBUG)

    httplib.debuglevel = 1
    httplib.HTTPConnection.debuglevel = 1

    urllib2_opener = urllib2.build_opener(
        urllib2.HTTPHandler(debuglevel=1),
        urllib2.HTTPSHandler(debuglevel=1)
    )
    urllib2.install_opener(urllib2_opener)


def parse_args():
    parser = argparse.ArgumentParser("Paysys monitorings engine")
    parser.add_argument(
        "--project",
        help="Defines project in ./configs (default: all project)",
        default="all",
    )
    parser.add_argument(
        "--environment",
        help="Defines environment in project (default: all environments)",
        default="all",
    )
    parser.add_argument(
        "--config",
        help="Defines config in environment (default: all configs)",
        default="all",
    )
    parser.add_argument(
        "--dry", help="Dry run, not applies changes in juggler",
        action="store_true"
    )
    parser.add_argument(
        "--no-cleanup", help="Do not remove entities, which are missing from configs.",
        action="store_true",
        dest="no_cleanup",
    )
    parser.add_argument(
        '-v', '--verbose', dest='verbose', action='store_true',
        help='Set log level to debug')

    args = parser.parse_args()

    if args.verbose:
        enable_debug()

    return args


def main():
    init_logger()
    options = parse_args()
    MonitoringsApplier(options).process_projects()
    sys.exit()


if __name__ == "__main__":
    main()
